% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Spectra-functions.R, R/Spectra.R
\name{processingLog}
\alias{processingLog}
\alias{scalePeaks}
\alias{addProcessing}
\alias{applyProcessing}
\alias{bin}
\alias{containsMz}
\alias{containsNeutralLoss}
\alias{entropy}
\alias{pickPeaks}
\alias{replaceIntensitiesBelow}
\alias{reset}
\alias{smooth}
\alias{spectrapply}
\alias{addProcessing,Spectra-method}
\alias{applyProcessing,Spectra-method}
\alias{bin,Spectra-method}
\alias{containsMz,Spectra-method}
\alias{containsNeutralLoss,Spectra-method}
\alias{entropy,Spectra-method}
\alias{entropy,ANY-method}
\alias{pickPeaks,Spectra-method}
\alias{replaceIntensitiesBelow,Spectra-method}
\alias{reset,Spectra-method}
\alias{smooth,Spectra-method}
\alias{spectrapply,Spectra-method}
\title{Data manipulation and analysis methods}
\usage{
processingLog(x)

scalePeaks(x, by = sum, msLevel. = uniqueMsLevels(x))

\S4method{addProcessing}{Spectra}(object, FUN, ..., spectraVariables = character())

\S4method{applyProcessing}{Spectra}(
  object,
  f = processingChunkFactor(object),
  BPPARAM = bpparam(),
  ...
)

\S4method{bin}{Spectra}(
  x,
  binSize = 1L,
  breaks = NULL,
  msLevel. = uniqueMsLevels(x),
  FUN = sum,
  zero.rm = TRUE
)

\S4method{containsMz}{Spectra}(
  object,
  mz = numeric(),
  tolerance = 0,
  ppm = 20,
  which = c("any", "all"),
  BPPARAM = bpparam()
)

\S4method{containsNeutralLoss}{Spectra}(
  object,
  neutralLoss = 0,
  tolerance = 0,
  ppm = 20,
  BPPARAM = bpparam()
)

\S4method{entropy}{Spectra}(object, normalized = TRUE)

\S4method{entropy}{ANY}(object, ...)

\S4method{pickPeaks}{Spectra}(
  object,
  halfWindowSize = 2L,
  method = c("MAD", "SuperSmoother"),
  snr = 0,
  k = 0L,
  descending = FALSE,
  threshold = 0,
  msLevel. = uniqueMsLevels(object),
  ...
)

\S4method{replaceIntensitiesBelow}{Spectra}(
  object,
  threshold = min,
  value = 0,
  msLevel. = uniqueMsLevels(object)
)

\S4method{reset}{Spectra}(object, ...)

\S4method{smooth}{Spectra}(
  x,
  halfWindowSize = 2L,
  method = c("MovingAverage", "WeightedMovingAverage", "SavitzkyGolay"),
  msLevel. = uniqueMsLevels(x),
  ...
)

\S4method{spectrapply}{Spectra}(
  object,
  FUN,
  ...,
  chunkSize = integer(),
  f = factor(),
  BPPARAM = SerialParam()
)
}
\arguments{
\item{x}{A \code{Spectra}.}

\item{by}{For \code{scalePeaks()}: function to calculate a single \code{numeric} from
intensity values of a spectrum by which all intensities (of
that spectrum) should be divided by. The default \code{by = sum} will
divide intensities of each spectrum by the sum of intensities of that
spectrum.}

\item{msLevel.}{\code{integer} defining the MS level(s) of the spectra to which
the function should be applied (defaults to all MS levels of \code{object}.}

\item{object}{A \code{Spectra} object.}

\item{FUN}{For \code{addProcessing()}: function to be applied to the peak matrix
of each spectrum in \code{object}.
For \code{bin()}: function to aggregate intensity values of peaks falling
into the same bin. Defaults to \code{FUN = sum} thus summing up intensities.
For \code{spectrapply()} and \code{chunkapply()}: function to be applied to
each individual or each chunk of \code{Spectra}.}

\item{...}{Additional arguments passed to internal and downstream functions.}

\item{spectraVariables}{For \code{addProcessing()}: \code{character} with additional
spectra variables that should be passed along to the function defined
with \code{FUN}. See function description for details.}

\item{f}{For \code{spectrapply()} and \code{applyProcessing()}: \code{factor} defining
how \code{object} should be splitted for eventual parallel processing.
Defaults to \code{factor()} for \code{spectrapply()} hence the object is not
splitted while it defaults to \code{f = processingChunkSize(object)} for
\code{applyProcessing()} splitting thus the object by default into chunks
depending on \code{\link[=processingChunkSize]{processingChunkSize()}}.}

\item{BPPARAM}{Parallel setup configuration. See \code{\link[BiocParallel:register]{BiocParallel::bpparam()}}
for more information. This is passed directly to the
\code{\link[=backendInitialize]{backendInitialize()}} method of the \linkS4class{MsBackend}. See also
\code{\link[=processingChunkSize]{processingChunkSize()}} for additional information on parallel
processing.}

\item{binSize}{For \code{bin()}: \code{numeric(1)} defining the size for the m/z bins.
Defaults to \code{binSize = 1}.}

\item{breaks}{For \code{bin()}: \code{numeric} defining the m/z breakpoints between
bins.}

\item{zero.rm}{For \code{bin()}: \code{logical(1)} indicating whether to remove bins
with zero intensity. Defaults to \code{TRUE}, meaning the function will
discard bins created with an intensity of 0 to enhance memory
efficiency.}

\item{mz}{For \code{containsMz()}: \code{numeric} with the m/z value(s) of the mass
peaks to check.}

\item{tolerance}{For \code{containsMz()} and \code{neutralLoss()}:
\code{numeric(1)} allowing to define a constant maximal accepted difference
between m/z values for peaks to be matched.}

\item{ppm}{For \code{containsMz()} and \code{neutralLoss()}: \code{numeric(1)} defining a
relative, m/z-dependent, maximal accepted difference between m/z values
for peaks to be matched.}

\item{which}{For \code{containsMz()}: either \code{"any"} or \code{"all"} defining whether
any (the default) or all provided \code{mz} have to be present in the
spectrum.}

\item{neutralLoss}{for \code{containsNeutralLoss()}: \code{numeric(1)} defining the
value which should be subtracted from the spectrum's precursor m/z.}

\item{normalized}{for \code{entropy()}: \code{logical(1)} whether the normalized
entropy should be calculated (default). See also
\code{\link[MsCoreUtils:entropy]{MsCoreUtils::nentropy()}} for details.}

\item{halfWindowSize}{For \code{pickPeaks()}: \code{integer(1)}, used in the
identification of the mass peaks: a local maximum has to be the
maximum in the window from \code{(i - halfWindowSize):(i + halfWindowSize)}.
For \code{smooth()}: \code{integer(1)}, used in the smoothing algorithm, the
window reaches from \code{(i - halfWindowSize):(i + halfWindowSize)}.}

\item{method}{For \code{pickPeaks()}: \code{character(1)}, the noise estimators that
should be used, currently the the \emph{M}edian \emph{A}bsolute \emph{D}eviation
(\code{method = "MAD"}) and Friedman's Super Smoother
(\code{method = "SuperSmoother"}) are supported.
For \code{smooth()}: \code{character(1)}, the smoothing function that should be
used, currently, the Moving-Average- (\code{method = "MovingAverage"}),
Weighted-Moving-Average- (\verb{method = "WeightedMovingAverage")},
Savitzky-Golay-Smoothing (\code{method = "SavitzkyGolay"}) are supported.}

\item{snr}{For \code{pickPeaks()}: \code{double(1)} defining the
\emph{S}ignal-to-\emph{N}oise-\emph{R}atio. The intensity of a local maximum has to be
higher than \code{snr * noise} to be considered as peak.}

\item{k}{For \code{pickPeaks()}: \code{integer(1)}, number of values left and right of
the peak that should be considered in the weighted mean calculation.}

\item{descending}{For \code{pickPeaks()}: \code{logical}, if \code{TRUE} just values
betwee the nearest valleys around the peak centroids are used.}

\item{threshold}{For \code{pickPeaks()}: a \code{numeric(1)} defining the proportion
of the maximal peak intensity. Only values above the threshold are
used for the weighted mean calculation.
For \code{replaceIntensitiesBelow()}: a \code{numeric(1)} defining the threshold
or a \code{function} to calculate the threshold for each spectrum on its
intensity values. Defaults to \code{threshold = min}.}

\item{value}{For \code{replaceIntensitiesBelow()}: \code{numeric(1)} defining the
value with which intensities should be replaced with.}

\item{chunkSize}{For \code{spectrapply()}: size of the chunks into which the
\code{Spectra} should be split. This parameter overrides parameters
\code{f} and \code{BPPARAM}.}
}
\value{
See the documentation of the individual functions for a description of the
return value.
}
\description{
Various data analysis functions are available for \code{Spectra} objects. These
can be categorized into functions that either return a \code{Spectra} object
(with the manipulated data) and functions that directly return the
result from the calculation. For the former category, the data manipulations
are cached in the result object's \emph{processing queue} and only exectuted
on-the-fly when the respective data gets extracted from the \code{Spectra} (see
section \emph{The processing queue} for more information).

For the second category, the calculations are directly executed and the
result, usually one value per spectrum, returned. Generally, to reduce
memory demand, a chunk-wise processing of the data is performed.
}
\section{Data analysis methods returning a \code{Spectra}}{


The methods listed here return a \code{Spectra} object as a result.
\itemize{
\item \code{addProcessing()}: adds an arbitrary function that should be applied to the
peaks matrix of every spectrum in \code{object}. The function (can be passed
with parameter \code{FUN}) is expected to take a peaks matrix as input and to
return a peaks matrix. A peaks matrix is a numeric matrix with two columns,
the first containing the m/z values of the peaks and the second the
corresponding intensities. The function has to have \code{...} in its
definition. Additional arguments can be passed with \code{...}. With parameter
\code{spectraVariables} it is possible to define additional spectra variables
from \code{object} that should be passed to the function \code{FUN}. These will be
passed by their name (e.g. specifying \code{spectraVariables = "precursorMz"}
will pass the spectra's precursor m/z as a parameter named \code{precursorMz}
to the function. The only exception is the spectra's MS level, these will
be passed to the function as a parameter called \code{spectrumMsLevel} (i.e.
with \code{spectraVariables = "msLevel"} the MS levels of each spectrum will be
submitted to the function as a parameter called \code{spectrumMsLevel}).
Examples are provided in the package vignette.
\item \code{bin()}: aggregates individual spectra into discrete (m/z) bins. Binning is
performed only on spectra of the specified MS level(s) (parameter
\code{msLevel}, by default all MS levels of \code{x}). The bins can be defined with
parameter \code{breaks} which by default are equally sized bins, with size
being defined by parameter \code{binSize}, from the minimal to the maximal m/z
of all spectra (of MS level \code{msLevel}) within \code{x}. The same bins are used
for all spectra in \code{x}. All intensity values for peaks falling into the
same bin are aggregated using the function provided with parameter \code{FUN}
(defaults to \code{FUN = sum}, i.e. all intensities are summed up). Note that
the binning operation is applied to the peak data on-the-fly upon data
access and it is possible to \emph{revert} the operation with the \code{reset()}
function (see description of \code{reset()} below).
\item \code{countIdentifications}: counts the number of identifications each scan has
led to. See \code{\link[=countIdentifications]{countIdentifications()}} for more details.
\item \code{pickPeaks()}: picks peaks on individual spectra using a moving
window-based approach (window size = \code{2 * halfWindowSize}). For noisy
spectra there are currently two different noise estimators available,
the \emph{M}edian \emph{A}bsolute \emph{D}eviation (\code{method = "MAD"}) and
Friedman's Super Smoother (\code{method = "SuperSmoother"}),
as implemented in the \code{\link[MsCoreUtils:noise]{MsCoreUtils::noise()}}.
The method supports also to optionally \emph{refine} the m/z value of
the identified centroids by considering data points that belong (most
likely) to the same mass peak. Therefore the m/z value is calculated as an
intensity weighted average of the m/z values within the peak region.
The peak region is defined as the m/z values (and their respective
intensities) of the \code{2 * k} closest signals to the centroid or the closest
valleys (\code{descending = TRUE}) in the \code{2 * k} region. For the latter the \code{k}
has to be chosen general larger. See \code{\link[MsCoreUtils:refineCentroids]{MsCoreUtils::refineCentroids()}} for
details.
If the ratio of the signal to the highest intensity of the peak is below
\code{threshold} it will be ignored for the weighted average.
\item \code{replaceIntensitiesBelow()}: replaces intensities below a specified
threshold with the provided \code{value}. Parameter \code{threshold} can be either
a single numeric value or a function which is applied to all non-\code{NA}
intensities of each spectrum to determine a threshold value for each
spectrum. The default is \code{threshold = min} which replaces all values
which are <= the minimum intensity in a spectrum with \code{value} (the
default for \code{value} is \code{0}). Note that the function specified with
\code{threshold} is expected to have a parameter \code{na.rm} since \code{na.rm = TRUE}
will be passed to the function. If the spectrum is in profile mode,
ranges of successive non-0 peaks <= \code{threshold} are set to 0.
Parameter \code{msLevel.} allows to apply this to only spectra of certain MS
level(s).
\item \code{scalePeaks()}: scales intensities of peaks within each spectrum depending
on parameter \code{by}. With \code{by = sum} (the default) peak intensities are
divided by the sum of peak intensities within each spectrum. The sum of
intensities is thus 1 for each spectrum after scaling. Parameter
\code{msLevel.} allows to apply the scaling of spectra of a certain MS level.
By default (\code{msLevel. = uniqueMsLevels(x)}) intensities for all
spectra will be scaled.
\item \code{smooth()}: smooths individual spectra using a moving window-based approach
(window size = \code{2 * halfWindowSize}). Currently, the
Moving-Average- (\code{method = "MovingAverage"}),
Weighted-Moving-Average- (\verb{method = "WeightedMovingAverage")},
weights depending on the distance of the center and calculated
\code{1/2^(-halfWindowSize:halfWindowSize)}) and
Savitzky-Golay-Smoothing (\code{method = "SavitzkyGolay"}) are supported.
For details how to choose the correct \code{halfWindowSize} please see
\code{\link[MsCoreUtils:smooth]{MsCoreUtils::smooth()}}.
}
}

\section{Data analysis methods returning the result from the calculation}{


The functions listed in this section return immediately the result from the
calculation. To reduce memory demand (and allow parallel processing) the
calculations a chunk-wise processing is generally performed.
\itemize{
\item \code{chunkapply()}: apply an arbitrary function to chunks of spectra. See
\code{\link[=chunkapply]{chunkapply()}} for details and examples.
\item \code{containsMz()}: checks for each of the spectra whether they contain mass
peaks with an m/z equal to \code{mz} (given acceptable difference as defined by
parameters \code{tolerance} and \code{ppm} - see \code{\link[MsCoreUtils:matching]{MsCoreUtils::common()}} for
details). Parameter
\code{which} allows to define whether any (\code{which = "any"}, the default) or
all (\code{which = "all"}) of the \code{mz} have to match. The function returns
\code{NA} if \code{mz} is of length 0 or is \code{NA}.
\item \code{containsNeutralLoss()}: checks for each spectrum in \code{object} if it has a
peak with an m/z value equal to its precursor m/z - \code{neutralLoss} (given
acceptable difference as defined by parameters \code{tolerance} and \code{ppm}).
Returns \code{NA} for MS1 spectra (or spectra without a precursor m/z).
\item \code{entropy()}: calculates the entropy of each spectra based on the metrics
suggested by Li et al. (https://doi.org/10.1038/s41592-021-01331-z).
See also \code{\link[MsCoreUtils:entropy]{MsCoreUtils::nentropy()}} in the \emph{MsCoreUtils} package for
details.
\item \code{estimatePrecursorIntensity()}: defines the precursor intensities for MS2
spectra using the intensity of the matching MS1 peak from the
closest MS1 spectrum (i.e. the last MS1 spectrum measured before the
respective MS2 spectrum). With \code{method = "interpolation"} it is also
possible to calculate the precursor intensity based on an interpolation of
intensity values (and retention times) of the matching MS1 peaks from the
previous and next MS1 spectrum. See \code{\link[=estimatePrecursorIntensity]{estimatePrecursorIntensity()}} for
examples and more details.
\item \code{estimatePrecursorMz()}: \strong{for DDA data}: allows to estimate a fragment
spectra's precursor m/z based on the reported precursor m/z and the data
from the previous MS1 spectrum. See \code{\link[=estimatePrecursorMz]{estimatePrecursorMz()}} for details.
\item \code{neutralLoss()}: calculates neutral loss spectra for fragment spectra. See
\code{\link[=neutralLoss]{neutralLoss()}} for detailed documentation.
\item \code{spectrapply()}: applies a given function to each individual spectrum or
sets of a \code{Spectra} object. By default, the \code{Spectra} is split into
individual spectra (i.e. \code{Spectra} of length 1) and the function \code{FUN}
is applied to each of them. An alternative splitting can be defined with
parameter \code{f}. Parameters for \code{FUN} can be passed using \code{...}.
The returned result and its order depend on the function \code{FUN} and how
\code{object} is split (hence on \code{f}, if provided). Parallel processing is
supported and can be configured with parameter \code{BPPARAM}, is however only
suggested for computational intense \code{FUN}.
As an alternative to the (eventual parallel) processing of the full
\code{Spectra}, \code{spectrapply()} supports also a chunk-wise processing. For this,
parameter \code{chunkSize} needs to be specified. \code{object} is then split into
chunks of size \code{chunkSize} which are then (stepwise) processed by \code{FUN}.
This guarantees a lower memory demand (especially for on-disk backends)
since only the data for one chunk needs to be loaded into memory in each
iteration. Note that by specifying \code{chunkSize}, parameters \code{f} and
\code{BPPARAM} will be ignored.
See also \code{chunkapply()} above or examples below for details on chunk-wise
processing.
}
}

\section{The processing queue}{


Operations that modify mass peak data, i.e. the m/z and intensity values of
a \code{Spectra} are generally not applied immediately to the data but are
\emph{cached} within the object's \emph{processing queue}. These operations are then
applied to the data only upon request, for example when m/z and/or
intensity values are extracted. This lazy execution guarantees that the
same functionality can be applied to any \code{Spectra} object, regardless of
the type of backend that is used. Thus, data manipulation operations can
also be applied to data that is \emph{read only}. As a side effect, this enables
also to \emph{undo} operations using the \code{reset()} function.

Functions related to the processing queue are:
\itemize{
\item \code{applyProcessing()}: for \code{Spectra} objects that use a \strong{writeable} backend
only: apply all steps from the lazy processing queue to the peak data and
write it back to the data storage. Parameter \code{f} allows to specify how
\code{object} should be split for parallel processing. This should either be
equal to the \code{dataStorage}, or \code{f = rep(1, length(object))} to disable
parallel processing alltogether. Other partitionings might result in
errors (especially if a \code{MsBackendHdf5Peaks} backend is used).
\item \code{processingLog()}: returns a \code{character} vector with the processing log
messages.
\item \code{reset()}: restores the data to its original state (as much as possible):
removes any processing steps from the lazy processing queue and calls
\code{reset()} on the backend which, depending on the backend, can also undo
e.g. data filtering operations. Note that a \verb{reset*(} call after
\code{applyProcessing()} will not have any effect. See examples below for more
information.
}
}

\examples{

## Load a `Spectra` object with LC-MS/MS data.
fl <- system.file("TripleTOF-SWATH", "PestMix1_DDA.mzML",
    package = "msdata")
sps_dda <- Spectra(fl)
sps_dda


##  --------  FUNCTIONS RETURNING A SPECTRA  --------

## Replace peak intensities below 40 with a value of 1
sps_mod <- replaceIntensitiesBelow(sps_dda, threshold = 20, value = 1)
sps_mod

## Get the intensities of the first spectrum before and after the
## operation
intensity(sps_dda[1])
intensity(sps_mod[1])

## Remove all peaks with an intensity below 5.
sps_mod <- filterIntensity(sps_dda, intensity = c(5, Inf))

intensity(sps_mod)

## In addition it is possible to pass a function to `filterIntensity()`: in
## the example below we want to keep only peaks that have an intensity which
## is larger than one third of the maximal peak intensity in that spectrum.
keep_peaks <- function(x, prop = 3) {
    x > max(x, na.rm = TRUE) / prop
}
sps_mod <- filterIntensity(sps_dda, intensity = keep_peaks)
intensity(sps_mod)

## We can also change the proportion by simply passing the `prop` parameter
## to the function. To keep only peaks that have an intensity which is
## larger than half of the maximum intensity:
sps_mod <- filterIntensity(sps_dda, intensity = keep_peaks, prop = 2)
intensity(sps_mod)

## With the `scalePeaks()` function we can alternatively scale the
## intensities of mass peaks per spectrum to relative intensities. This
## is specifically useful for fragment (MS2) spectra. We below thus
## scale the intensities per spectrum by the total sum of intensities
## (such that the sum of all intensities per spectrum is 1).
## Below we scale the intensities of all MS2 spectra in our data set.
sps_mod <- scalePeaks(sps_dda, msLevel = 2L)

## MS1 spectra were not affected
sps_mod |>
    filterMsLevel(1L) |>
    intensity()

## Intensities of MS2 spectra were scaled
sps_mod |>
    filterMsLevel(2L) |>
    intensity()

## Since data manipulation operations are by default not directly applied to
## the data but only cached in the internal processing queue, it is also
## possible to remove these data manipulations with the `reset()` function:
tmp <- reset(sps_mod)
tmp
lengths(sps_dda) |> head()
lengths(sps_mod) |> head()
lengths(tmp) |> head()

## Data manipulation operations cached in the processing queue can also be
## applied to the mass peaks data with the `applyProcessing()` function, if
## the `Spectra` uses a backend that supports that (i.e. allows replacing
## the mass peaks data). Below we first change the backend to a
## `MsBackendMemory()` and then use the `applyProcessing()` to modify the
## mass peaks data
sps_dda <- setBackend(sps_dda, MsBackendMemory())
sps_mod <- filterIntensity(sps_dda, intensity = c(5, Inf))
sps_mod <- applyProcessing(sps_mod)
sps_mod

## While we can't *undo* this filtering operation now using the `reset()`
## function, accessing the data would now be faster, because the operation
## does no longer to be applied to the original data before returning to the
## user.


##  --------  FUNCTIONS RETURNING THE RESULT  --------

## With the `spectrapply()` function it is possible to apply an
## arbitrary function to each spectrum in a Spectra.
## In the example below we calculate the mean intensity for each spectrum
## in a subset of the sciex_im data. Note that we can access all variables
## of each individual spectrum either with the `$` operator or the
## corresponding method.
res <- spectrapply(sps_dda[1:20], FUN = function(x) mean(x$intensity[[1]]))
head(res)

## As an alternative, applying a function `FUN` to a `Spectra` can be
## performed *chunk-wise*. The advantage of this is, that only the data for
## one chunk at a time needs to be loaded into memory reducing the memory
## demand. This type of processing can be performed by specifying the size
## of the chunks (i.e. number of spectra per chunk) with the `chunkSize`
## parameter
spectrapply(sps_dda[1:20], lengths, chunkSize = 5L)

## Precursor intensity estimation. Some manufacturers don't report the
## precursor intensity for MS2 spectra:
sps_dda |>
    filterMsLevel(2L) |>
    precursorIntensity()

## This intensity can however be estimated from the previously measured
## MS1 scan with the `estimatePrecursorIntensity()` function:
pi <- estimatePrecursorIntensity(sps_dda)

## This function returned the result as a `numeric` vector with one
## value per spectrum:
pi

## We can replace the precursor intensity values of the originating
## object:
sps_dda$precursorIntensity <- pi
sps_dda |>
    filterMsLevel(2L) |>
    precursorIntensity()

}
\seealso{
\itemize{
\item \code{\link[=compareSpectra]{compareSpectra()}} for calculation of spectra similarity scores.
\item \code{\link[=processingChunkSize]{processingChunkSize()}} for information on parallel and chunk-wise data
processing.
\item \link{Spectra} for a general description of the \code{Spectra} object.
}
}
\author{
Sebastian Gibb, Johannes Rainer, Laurent Gatto, Philippine Louail, Nir Shahaf, Mar Garcia-Aloy
}
