#' TSAR Condition IDs
#'
#' This function is used to extract information of the condition IDs from a
#'     loaded TSA Analysis Data file. Condition IDs are automatically generated
#'     by the read_analysis function in the automated workflow. This returns
#'     either a character vector of unique IDs present or a numeric value of the
#'     number of unique IDs.
#'
#'
#' @param analysis_data a data frame that is unmerged and generated by
#'     TSAR::read_analysis() or a merged TSA data frame generated by
#'     TSAR::merge_TSA(). Data frames require a column named 'condition_ID'.
#' @param n logical value; \code{n = FALSE} by default. When TRUE, a numeric
#'     value of unique IDs is returned. When FALSE, a character vector of unique
#'     IDs are returned.
#'
#' @return Either a character vector of condition_IDs or a numeric value.
#'
#' @examples
#' data("example_tsar_data")
#' condition_IDs(example_tsar_data)
#'
#' @family TSA Summary Functions
#' @seealso \code{\link{merge_TSA}} and \code{\link{read_analysis}}
#' for preparing input.
#' @export
#'
condition_IDs <- function(
    analysis_data, # analysis data to determine number of unique conditions
    n = FALSE # c("T", "F") F = vector, T = number summary
    ) {
    if (n) {
        return(length(unique(analysis_data$condition_ID)))
    } else {
        return(unique(analysis_data$condition_ID))
    }
}




#' TSAR Well IDs
#'
#' This function is used to extract information of the well IDs from a
#'     merged TSA data frame. Well IDs are automatically generated
#'     by the read_analysis and read_raw_data functions in the automated
#'     workflow. This function returns
#'     either a character vector of unique IDs present or a numeric value of the
#'     number of unique IDs.
#'
#'
#' @param tsa_data a data frame that is merged and generated by
#'     TSAR::merge_TSA(), or an unmerged data frame read by
#'     TSAR::read_analysis() or TSAR::read_raw_data().
#'     Data frames require a column named 'well_ID'.
#' @param n logical value; \code{n = FALSE} by default. When TRUE, a numeric
#'     value of unique IDs is returned. When FALSE, a character vector of unique
#'     IDs are returned.
#'
#' @return Either a character vector of unique well IDs or a numeric value.
#'
#' @family TSA Summary Functions
#' @seealso \code{\link{merge_TSA}}, \code{\link{read_raw_data}},
#' and \code{\link{read_analysis}} for preparing input.
#'
#' @examples
#' data("example_tsar_data")
#' well_IDs(example_tsar_data)
#'
#' @export

well_IDs <- function(
    tsa_data, # tsa data to determine number of unique conditions
    n = FALSE # c("T", "F") F = List, T = number summary
    ) {
    if (n) {
        return(length(unique(tsa_data$well_ID)))
    } else {
        return(unique(tsa_data$well_ID))
    }
}

#' TSA Proteins
#'
#' This function is used to extract information from a data frame of TSA data.
#'     The Protein values should be assigned in the TSA software.
#'
#' @param tsa_data a data frame that is merged and generated by
#'     TSAR::merge_TSA(), or an unmerged data frame read by
#'     TSAR::read_analysis() or TSAR::read_raw_data().
#'     The data frame must have a column named 'Protein'.
#' @param n logical value; \code{n = FALSE} by default. When TRUE, a numeric
#'     value describing the number of unique protein names is returned.
#'     When FALSE, a character vector of unique IDs are returned.
#'
#' @return Either a character vector of unique well_IDs or a numeric value.
#'
#' @family TSA Summary Functions
#' @seealso \code{\link{merge_TSA}}, \code{\link{read_raw_data}},
#'     and \code{\link{read_analysis}} for preparing input.
#'
#' @examples
#' data("example_tsar_data")
#' TSA_proteins(example_tsar_data)
#'
#' @export

TSA_proteins <- function(
    tsa_data, # tsa data to determine number of unique conditions
    n = FALSE # c("T", "F") F = List, T = number summary
    ) {
    if (n) {
        return(length(levels(factor(tsa_data$Protein))))
    } else {
        return(levels(factor(tsa_data$Protein)))
    }
}

#' TSA Ligands
#'
#' This function is used to extract information from a data frame of TSA data.
#'     The Ligand values should be assigned in the TSA software.
#'
#' @param tsa_data a data frame that is merged and generated by
#'     TSAR::merge_TSA(), or an unmerged data frame read by
#'     TSAR::read_analysis() or TSAR::read_raw_data().
#'     The data frame must have a column named 'Ligand'.
#' @param n logical value; \code{n = FALSE} by default. When TRUE, a numeric
#'     value describing the number of unique ligand names is returned.
#'     When FALSE, a character vector of unique IDs are returned.
#' @return Either a character vector of unique well_IDs or a numeric value.
#' @family TSA Summary Functions
#' @seealso \code{\link{merge_TSA}}, \code{\link{read_raw_data}},
#'     and \code{\link{read_analysis}} for preparing input.
#'
#' @examples
#' data("example_tsar_data")
#' TSA_ligands(example_tsar_data)
#'
#' @export

TSA_ligands <- function(
    tsa_data, # tsa data to determine number of unique conditions
    n = FALSE # c("T", "F") F = List, T = number summary
    ) {
    if (n) {
        return(length(levels(factor(tsa_data$Ligand))))
    } else {
        return(levels(factor(tsa_data$Ligand)))
    }
}
