\name{plotCat}

\alias{plotCat}

\title{Plotting correspondence at the top curves}

\description{
  This function plots corresponding at the top (CAT) curves
  using overlap proportions computed by \code{computeCat}.
  A number of arguments can be used for a pretty display, and for
  annotating the plot, and adding the legend
}

\usage{
plotCat(catData, whichToPlot = 1:length(catData),
     preComputedPI, size=500, main="CAT-plot",
     minYlim=0, maxYlim=1, col, pch, lty, cex=1, lwd=1,
     spacePts=10, cexPts=1, legend=TRUE, legendText,
     where="center", legCex=1,
     plotLayout=layout(matrix(1:2, ncol = 2, byrow = TRUE), widths = c(0.7, 0.3)), ...) 
}

\arguments{
  \item{catData}{The ouput list obtained from \code{computeCat},
    containing the overlapping proportions among pairs of
    ordered vectors.  Names in \code{catData} are used for
    annotating the legend if \code{legendText} is not provided
    (see below).}

  \item{whichToPlot}{numeric vector. Indexes corresponding
    to the elements of \code{catData} to be selected for
    displaying in the plot.}

  \item{preComputedPI}{numeric matrix. Probability intervals
    computed using the \code{calcHypPI} function.
    It is used to add grey shades to the plot corresponding
    to overlapping proportion probabilities based on the
    hypegeomtric distribution. If missing no PI will be
    added to the plot.}

  \item{size}{numeric. The number of top ranking features
    to be displayed in the plot.}

  \item{main}{character. The title of the plot, if not provided,
    \code{main} default is "CAT-plot".}

  \item{minYlim}{numeric. The lower numeric value of the y axis,
    to be displayed in the plot.}

  \item{maxYlim}{numeric. The upper numeric value of the y axis,
    to be displayed in the plot.}

  \item{col}{character or numeric. Vector specifying colors
    for CAT curves  plotting. \code{col} default uses
    \code{rainbow} function to generate a color vector
    for all CAT curves in \code{catData}.
    When provided by the user, it will be recycled if needed.}

  \item{pch}{graphical parameter. \code{pch} specifies point
    types  for annotating the CAT curves.
    If not provided, \code{pch} is created by default, and recycled
    if needed. See \code{par} for details.}

  \item{lty}{graphical parameter. The type of line for the plot.
    If not provided generated by default, recycled id needed.
    See \code{par} if needed.}

  \item{cex}{numeric. Standard graphical parameter useful
    for controlling axes and title annotation size.
    See \code{par}.}

 \item{lwd}{numeric. Standard graphical parameter useful
   for controlling line size. See \code{par}.}

 \item{spacePts}{numeric. Specifies the interval to be used for
   adding point labels on the CAT curves (evenly spaced along
   the x axis dimention).}

  \item{cexPts}{numeric. Graphical parameter useful for controlling
   points size used for annotating CAT-plot lines.}
 
 \item{legend}{logical. Wheter a legend should be added to the plot.}

 \item{legendText}{character. A vector used for legend creation.
   \code{legendText} default correspond to \code{catData} names.}

 \item{where}{character. The position of the plot where the legend
   will be created; \code{where} default is \code{'center'},
   see \code{legend}.}

   \item{legCex}{numeric. Graphical parameter setting
     the font size for the legend text.}

 \item{plotLayout}{A layout matrix to arrange the plot
   and the legend. For further details see \code{layout}.}

 \item{\dots}{Other graphical parameters, currently passed
   only to \code{legend} (e.g. the number of columns to be used
   in the legend, or the legend background).}
}

\details{
  This function uses outputs from \code{computeCat}
  and \code{calcHypPI} to plot the CAT curves and
  add grey shades corresponding to probability intervals.
  The default plot uses a pre-specified layout
  with separate areas for the plot and the legend.
  If not specified by the user, different points, colors and line
  types are used for the different CAT curves.
  If the CAT curves where computed using equal ranks
  (e.g. "equalRank" was passed to the \code{method}
  argument of the \code{computeCat} function),
  the user has the option of adding probability intervals
  to the plot. Such intervals must be pre-computed
  using the \code{calcHypPI} function.
}

\value{
  Produces an annotated CAT plot.
}


\seealso{
  See \code{\link{computeCat}}, \code{\link{calcHypPI}},
  \code{\link[grDevices]{rainbow}}, \code{\link[graphics]{par}},
  \code{\link[graphics]{legend}}, and \code{\link[graphics]{layout}}.
}


\note{
In order to make the "best looking" plot for your needs
you must play around with graphical parameters
}


\references{ 
  Irizarry, R. A.; Warren, D.; Spencer, F.; Kim, I. F.; Biswal, S.;
  Frank, B. C.; Gabrielson, E.; Garcia, J. G. N.; Geoghegan, J.;
  Germino, G.; Griffin, C.; Hilmer, S. C.; Hoffman, E.;
  Jedlicka, A. E.; Kawasaki, E.; Martinez-Murillo, F.;
  Morsberger, L.; Lee, H.; Petersen, D.; Quackenbush, J.;
  Scott, A.; Wilson, M.; Yang, Y.; Ye, S. Q.
  and Yu, W. Multiple-laboratory comparison of microarray platforms.
  Nat Methods, 2005, 2, 345-350

  Ross, A. E.; Marchionni, L.; Vuica-Ross, M.; Cheadle, C.;
  Fan, J.; Berman, D. M.; and Schaeffer E. M.
  Gene Expression Pathways of High Grade Localized Prostate Cancer.
  Prostate, 2011, 71, 1568-1578

  Benassi, B.; Flavin, R.; Marchionni, L.; Zanata, S.; Pan, Y.;
  Chowdhury, D.; Marani, M.; Strano, S.; Muti, P.; and Blandino, G.
  c-Myc is activated via USP2a-mediated modulation of microRNAs
  in prostate cancer. Cancer Discovery, 2012, March, 2, 236-247
  
}


\author{ Luigi Marchionni \email{marchion@jhu.edu}}


\examples{
###load data
data(matchBoxExpression)

###the column name for the identifiers and  the ranking statistics
idCol <- "SYMBOL"
byCol <- "t"

####filter the redundant features using SYMBOL and t-statistics
matchBoxExpression <- lapply(matchBoxExpression, filterRedundant, idCol=idCol, byCol=byCol)

###select and merge into a matrix
mat <- mergeData(matchBoxExpression, idCol=idCol, byCol=byCol)

###COMPUTE CAT
cpH2L <- computeCat(mat, idCol=1, size=round(nrow(mat)/1),
	decreasing=TRUE, method="equalRank")

###CATplot without probability intervals
par(mar=c(3,3,2,1))
plotCat(cpH2L, main="CAT-plot, decreasing t-statistics",
	cex=1, lwd=2, cexPts=1.5, spacePts=15,
	legend=TRUE, where="center",
	legCex=1, ncol=1)

###compute probability intervals
confInt <- calcHypPI(data=mat)

###CATplot with probability intervals
par(mar=c(3,3,2,1))
plotCat(cpH2L, main="CAT-plot, decreasing t-statistics, probability intevals",
	cex=1, lwd=2, cexPts=1.5, spacePts=15,
	legend=TRUE, where="center",
	legCex=1, ncol=1) 
}

\keyword{ manip}
