% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateGLV.R
\name{simulateGLV}
\alias{simulateGLV}
\title{Generalized Lotka-Volterra (gLV) simulation}
\usage{
simulateGLV(
  n_species,
  names_species = NULL,
  A = NULL,
  x0 = NULL,
  growth_rates = NULL,
  sigma_drift = 0.001,
  sigma_epoch = 0.1,
  sigma_external = 0.3,
  sigma_migration = 0.01,
  epoch_p = 0.001,
  t_external_events = NULL,
  t_external_durations = NULL,
  stochastic = TRUE,
  migration_p = 0.01,
  metacommunity_probability = NULL,
  error_variance = 0,
  norm = FALSE,
  t_end = 1000,
  ...
)
}
\arguments{
\item{n_species}{Integer: number of species}

\item{names_species}{Character: names of species. If NULL,
\code{paste0("sp", seq_len(n_species))} is used.
(default: \code{names_species = NULL})}

\item{A}{matrix: interaction matrix defining the positive and negative
interactions between n_species. If NULL, \code{randomA(n_species)} is used.
(default: \code{A = NULL})}

\item{x0}{Numeric: initial abundances of simulated species. If NULL,
\code{runif(n = n_species, min = 0, max = 1)} is used.
(default: \code{x0 = NULL})}

\item{growth_rates}{Numeric: growth rates of simulated species. If NULL,
\code{runif(n = n_species, min = 0, max = 1)} is used.
(default: \code{growth_rates = NULL})}

\item{sigma_drift}{Numeric: standard deviation of a normally distributed
noise applied in each time step (t_step)
(default: \code{sigma_drift = 0.001})}

\item{sigma_epoch}{Numeric: standard deviation of a normally distributed
noise applied to random periods of the community composition with frequency
defined by the epoch_p parameter
(default: \code{sigma_epoch = 0.1})}

\item{sigma_external}{Numeric: standard deviation of a normally distributed
noise applied to user-defined external events/disturbances
(default: \code{sigma_external = 0.3})}

\item{sigma_migration}{Numeric: standard deviation of a normally distributed
variable that defines the intensity of migration at each time step (t_step)
(default: \code{sigma_migration = 0.01})}

\item{epoch_p}{Numeric: the probability/frequency of random periodic
changes introduced to the community composition
(default: \code{epoch_p = 0.001})}

\item{t_external_events}{Numeric: the starting time points of defined
external events that introduce random changes to the community composition
(default: \code{t_external_events = NULL})}

\item{t_external_durations}{Numeric: respective duration of the external
events that are defined in the 't_external_events' (times) and
sigma_external (std).
(default: \code{t_external_durations = NULL})}

\item{stochastic}{Logical: whether to introduce noise in the simulation.
If False, sigma_drift, sigma_epoch, and sigma_external are ignored.
(default: \code{stochastic = FALSE})}

\item{migration_p}{Numeric: the probability/frequency of migration from a
metacommunity.
(default: \code{migration_p = 0.01})}

\item{metacommunity_probability}{Numeric: Normalized probability distribution
of the likelihood that species from the metacommunity can enter the community
during the simulation. If NULL, \code{rdirichlet(1, alpha = rep(1,n_species))} is
used.
(default: \code{metacommunity_probability = NULL})}

\item{error_variance}{Numeric: the variance of measurement error.
By default it equals to 0, indicating that the result won't contain any
measurement error. This value should be non-negative.
(default: \code{error_variance = 0})}

\item{norm}{Logical: whether the time series should be returned with
the abundances as proportions (\code{norm = TRUE}) or
the raw counts (default: \code{norm = FALSE})
(default: \code{norm = FALSE})}

\item{t_end}{Numeric: the end time of the simulationTimes, defining the
modeled time length of the community.
(default: \code{t_end = 1000})}

\item{...}{additional parameters, see \code{\link{utils}} to know more.}
}
\value{
\code{simulateGLV} returns a TreeSummarizedExperiment class object
}
\description{
Simulates time series with the generalized Lotka-Volterra model.
}
\details{
Simulates a community time series using the generalized Lotka-Volterra model,
defined as dx/dt = x(b+Ax), where x is the vector of species abundances,
diag(x) is a diagonal matrix with the diagonal values set to x.
A is the interaction matrix and b is the vector of growth rates.
}
\examples{

# generate a random interaction matrix
ExampleA <- randomA(n_species = 4, diagonal = -1)

# run the model with default values (only stochastic migration considered)
tse <- simulateGLV(n_species = 4, A = ExampleA)

# run the model with two external disturbances at time points 240 and 480
# with durations equal to 1 (10 time steps when t_step by default is 0.1).
ExampleGLV <- simulateGLV(
    n_species = 4, A = ExampleA,
    t_external_events = c(0, 240, 480), t_external_durations = c(0, 1, 1)
)

# run the model with no perturbation nor migration
set.seed(42)
tse1 <- simulateGLV(
    n_species = 4, A = ExampleA, stochastic = FALSE,
    sigma_migration = 0
)

# run the model with no perturbation nor migration but with measurement error
set.seed(42)
tse2 <- simulateGLV(
    n_species = 4, A = ExampleA, stochastic = FALSE,
    error_variance = 0.001, sigma_migration = 0
)

}
