% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculateUnifrac.R
\name{calculateUnifrac}
\alias{calculateUnifrac}
\alias{calculateUnifrac,ANY,phylo-method}
\alias{calculateUnifrac,TreeSummarizedExperiment,missing-method}
\alias{runUnifrac}
\title{Calculate weighted or unweighted (Fast) Unifrac distance}
\usage{
calculateUnifrac(x, tree, ...)

\S4method{calculateUnifrac}{ANY,phylo}(
  x,
  tree,
  weighted = FALSE,
  normalized = TRUE,
  BPPARAM = SerialParam(),
  ...
)

\S4method{calculateUnifrac}{TreeSummarizedExperiment,missing}(
  x,
  assay.type = assay_name,
  assay_name = exprs_values,
  exprs_values = "counts",
  tree_name = "phylo",
  transposed = FALSE,
  ...
)

runUnifrac(
  x,
  tree,
  weighted = FALSE,
  normalized = TRUE,
  nodeLab = NULL,
  BPPARAM = SerialParam(),
  ...
)
}
\arguments{
\item{x}{a numeric matrix or a
\code{\link[TreeSummarizedExperiment:TreeSummarizedExperiment-class]{TreeSummarizedExperiment}}
object containing a tree.

Please  note that \code{runUnifrac} expects a matrix with samples per row
and not per column. This is implemented to be compatible with other
distance calculations such as \code{\link[stats:dist]{dist}} as much as
possible.}

\item{tree}{if \code{x} is a matrix, a
\code{\link[TreeSummarizedExperiment:phylo]{phylo}} object matching the
matrix. This means that the phylo object and the columns should relate
to the same type of features (aka. microorganisms).}

\item{...}{optional arguments not used.}

\item{weighted}{\code{TRUE} or \code{FALSE}: Should use weighted-Unifrac
calculation? Weighted-Unifrac takes into account the relative abundance of
species/taxa shared between samples, whereas unweighted-Unifrac only
considers presence/absence. Default is \code{FALSE}, meaning the
unweighted-Unifrac distance is calculated for all pairs of samples.}

\item{normalized}{\code{TRUE} or \code{FALSE}: Should the output be
normalized such that values range from 0 to 1 independent of branch length
values? Default is \code{TRUE}. Note that (unweighted) \code{Unifrac} is
always normalized by total branch-length, and so this value is ignored when
\code{weighted == FALSE}.}

\item{BPPARAM}{A
\code{\link[BiocParallel:BiocParallelParam-class]{BiocParallelParam}}
object specifying whether the Unifrac calculation should be parallelized.}

\item{assay.type}{a single \code{character} value for specifying which
assay to use for calculation.}

\item{assay_name}{a single \code{character} value for specifying which
assay to use for calculation.
(Please use \code{assay.type} instead. At some point \code{assay_name}
will be disabled.)}

\item{exprs_values}{a single \code{character} value for specifying which
assay to use for calculation.
(Please use \code{assay.type} instead.)}

\item{tree_name}{a single \code{character} value for specifying which
tree will be used in calculation.
(By default: \code{tree_name = "phylo"})}

\item{transposed}{Logical scalar, is x transposed with cells in rows, i.e.,
is Unifrac distance calculated based on rows (FALSE) or columns (TRUE).
(By default: \code{transposed = FALSE})}

\item{nodeLab}{if \code{x} is a matrix,
a \code{character} vector specifying links between rows/columns and tips of \code{tree}.
The length must equal the number of rows/columns of \code{x}. Furthermore, all the
node labs must be present in \code{tree}.}
}
\value{
a sample-by-sample distance matrix, suitable for NMDS, etc.
}
\description{
This function calculates the (Fast) Unifrac distance for all sample-pairs
in a \code{\link[TreeSummarizedExperiment:TreeSummarizedExperiment-class]{TreeSummarizedExperiment}}
object.
}
\details{
Please note that if \code{calculateUnifrac} is used as a \code{FUN} for
\code{runMDS}, the argument \code{ntop} has to be set to \code{nrow(x)}.
}
\examples{
data(esophagus)
library(scater)
calculateUnifrac(esophagus, weighted = FALSE)
calculateUnifrac(esophagus, weighted = TRUE)
calculateUnifrac(esophagus, weighted = TRUE, normalized = FALSE)
# for using calculateUnifrac in conjunction with runMDS the tree argument
# has to be given separately. In addition, subsetting using ntop must
# be disabled
esophagus <- runMDS(esophagus, FUN = calculateUnifrac, name = "Unifrac",
                    tree = rowTree(esophagus),
                    exprs_values = "counts",
                    ntop = nrow(esophagus))
reducedDim(esophagus)
}
\references{
\url{http://bmf.colorado.edu/unifrac/}

The main implementation (Fast Unifrac) is adapted from the algorithm's
description in:

Hamady, Lozupone, and Knight,
``\href{http://www.nature.com/ismej/journal/v4/n1/full/ismej200997a.html}{Fast
UniFrac:} facilitating high-throughput phylogenetic analyses of microbial
communities including analysis of pyrosequencing and PhyloChip data.'' The
ISME Journal (2010) 4, 17--27.

See also additional descriptions of Unifrac in the following articles:

Lozupone, Hamady and Knight, ``Unifrac - An Online Tool for Comparing
Microbial Community Diversity in a Phylogenetic Context.'', BMC
Bioinformatics 2006, 7:371

Lozupone, Hamady, Kelley and Knight, ``Quantitative and qualitative (beta)
diversity measures lead to different insights into factors that structure
microbial communities.'' Appl Environ Microbiol. 2007

Lozupone C, Knight R. ``Unifrac: a new phylogenetic method for comparing
microbial communities.'' Appl Environ Microbiol. 2005 71 (12):8228-35.
}
\author{
Paul J. McMurdie.
Adapted for mia by Felix G.M. Ernst
}
