% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimateDiversity.R
\name{estimateDiversity}
\alias{estimateDiversity}
\alias{estimateDiversity,SummarizedExperiment-method}
\alias{estimateDiversity,TreeSummarizedExperiment-method}
\alias{estimateFaith}
\alias{estimateFaith,SummarizedExperiment,phylo-method}
\alias{estimateFaith,TreeSummarizedExperiment,missing-method}
\title{Estimate (alpha) diversity measures}
\usage{
estimateDiversity(
  x,
  assay.type = "counts",
  assay_name = NULL,
  index = c("coverage", "fisher", "gini_simpson", "inverse_simpson",
    "log_modulo_skewness", "shannon"),
  name = index,
  ...
)

\S4method{estimateDiversity}{SummarizedExperiment}(
  x,
  assay.type = "counts",
  assay_name = NULL,
  index = c("coverage", "fisher", "gini_simpson", "inverse_simpson",
    "log_modulo_skewness", "shannon"),
  name = index,
  ...,
  BPPARAM = SerialParam()
)

\S4method{estimateDiversity}{TreeSummarizedExperiment}(
  x,
  assay.type = "counts",
  assay_name = NULL,
  index = c("coverage", "faith", "fisher", "gini_simpson", "inverse_simpson",
    "log_modulo_skewness", "shannon"),
  name = index,
  tree_name = "phylo",
  ...,
  BPPARAM = SerialParam()
)

estimateFaith(
  x,
  tree = "missing",
  assay.type = "counts",
  assay_name = NULL,
  name = "faith",
  ...
)

\S4method{estimateFaith}{SummarizedExperiment,phylo}(
  x,
  tree,
  assay.type = "counts",
  assay_name = NULL,
  name = "faith",
  node_lab = NULL,
  ...
)

\S4method{estimateFaith}{TreeSummarizedExperiment,missing}(
  x,
  assay.type = "counts",
  assay_name = NULL,
  name = "faith",
  tree_name = "phylo",
  ...
)
}
\arguments{
\item{x}{a \code{\link{SummarizedExperiment}} object or \code{\link{TreeSummarizedExperiment}}.
The latter is recommended for microbiome data sets and tree-based alpha diversity indices.}

\item{assay.type}{the name of the assay used for
calculation of the sample-wise estimates.}

\item{assay_name}{a single \code{character} value for specifying which
assay to use for calculation.
(Please use \code{assay.type} instead. At some point \code{assay_name}
will be disabled.)}

\item{index}{a \code{character} vector, specifying the diversity measures
to be calculated.}

\item{name}{a name for the column(s) of the colData the results should be
stored in. By default this will use the original names of the calculated
indices.}

\item{...}{optional arguments:
\itemize{
\item{threshold}{ A numeric value in the unit interval,
determining the threshold for coverage index. By default,
\code{threshold} is 0.9.}
\item{quantile}{ Arithmetic abundance classes are evenly cut up to to
this quantile of the data. The assumption is that abundances higher than
this are not common, and they are classified in their own group.
By default, \code{quantile} is 0.5.}
\item{num_of_classes}{ The number of arithmetic abundance classes
from zero to the quantile cutoff indicated by \code{quantile}.
By default, \code{num_of_classes} is 50.}
\item{only.tips}{ A boolean value specifying whether to remove internal
nodes when Faith's index is calculated. When \code{only.tips=TRUE}, those
rows that are not tips of tree are removed.
(By default: \code{only.tips=FALSE})}
}}

\item{BPPARAM}{A
\code{\link[BiocParallel:BiocParallelParam-class]{BiocParallelParam}}
object specifying whether calculation of estimates should be parallelized.}

\item{tree_name}{a single \code{character} value for specifying which
rowTree will be used to calculate faith index.
(By default: \code{tree_name = "phylo"})}

\item{tree}{A phylogenetic tree that is used to calculate 'faith' index.
If \code{x} is a \code{TreeSummarizedExperiment}, \code{rowTree(x)} is
used by default.}

\item{node_lab}{NULL or a character vector specifying the links between rows and
node labels of \code{tree}. If a certain row is not linked with the tree, missing
instance should be noted as NA. When NULL, all the rownames should be found from
the tree. (By default: \code{node_lab = NULL})}
}
\value{
\code{x} with additional \code{\link{colData}} named \code{*name*}
}
\description{
Several functions for calculating (alpha) diversity indices, including
the \code{vegan} package options and some others.
}
\details{
The available indices include the \sQuote{Coverage},
\sQuote{Faith's phylogenetic diversity}, \sQuote{Fisher alpha},
\sQuote{Gini-Simpson},
\sQuote{Inverse Simpson}, \sQuote{log-modulo skewness}, and \sQuote{Shannon}
indices. See details for more information and references.

Alpha diversity is a joint quantity that combines elements or community richness
and evenness. Diversity increases, in general, when species richness or
evenness increase.

By default, this function returns all indices.

\itemize{

\item{'coverage' }{Number of species needed to cover a given fraction of
the ecosystem (50 percent by default). Tune this with the threshold
argument.}

\item{'faith' }{Faith's phylogenetic alpha diversity index measures how
long the taxonomic distance is between taxa that are present in the sample.
Larger values represent higher diversity. Using this index requires
rowTree. (Faith 1992)

If the data includes features that are not in tree's tips but in
internal nodes, there are two options. First, you can keep those features,
and prune the tree to match features so that each tip can be found from
the features. Other option is to remove all features that are not tips.
(See \code{only.tips} parameter)}

\item{'fisher' }{Fisher's alpha; as implemented in
\code{\link[vegan:diversity]{vegan::fisher.alpha}}. (Fisher et al. 1943)}

\item{'gini_simpson' }{Gini-Simpson diversity i.e. \eqn{1 - lambda},
where \eqn{lambda} is the
Simpson index, calculated as the sum of squared relative abundances.
This corresponds to the diversity index
'simpson' in \code{\link[vegan:diversity]{vegan::diversity}}.
This is also called Gibbs–Martin, or Blau index in sociology,
psychology and management studies. The Gini-Simpson index (1-lambda)
should not be
confused with Simpson's dominance (lambda), Gini index, or
inverse Simpson index (1/lambda).}

\item{'inverse_simpson' }{Inverse Simpson diversity:
\eqn{1/lambda} where \eqn{lambda=sum(p^2)} and p refers to relative
abundances.
This corresponds to the diversity index
'invsimpson' in vegan::diversity. Don't confuse this with the
closely related Gini-Simpson index}

\item{'log_modulo_skewness' }{The rarity index characterizes the
concentration of species at low abundance. Here, we use the skewness of
the frequency
distribution of arithmetic abundance classes (see Magurran & McGill 2011).
These are typically right-skewed; to avoid taking log of occasional
negative skews, we follow Locey & Lennon (2016) and use the log-modulo
transformation that adds a value of one to each measure of skewness to
allow logarithmization.}

\item{'shannon' }{Shannon diversity (entropy).}

}
}
\examples{
data(GlobalPatterns)
tse <- GlobalPatterns

# All index names as known by the function
index <- c("shannon","gini_simpson","inverse_simpson", "coverage", "fisher", 
"faith",  "log_modulo_skewness")

# Corresponding polished names
name <- c("Shannon","GiniSimpson","InverseSimpson", "Coverage", "Fisher", 
"Faith",  "LogModSkewness")

# Calculate diversities
tse <- estimateDiversity(tse, index = index)

# The colData contains the indices with their code names by default
colData(tse)[, index]

# Removing indices
colData(tse)[, index] <- NULL

# 'threshold' can be used to determine threshold for 'coverage' index
tse <- estimateDiversity(tse, index = "coverage", threshold = 0.75)
# 'quantile' and 'num_of_classes' can be used when
# 'log_modulo_skewness' is calculated
tse <- estimateDiversity(tse, index = "log_modulo_skewness",
       quantile = 0.75, num_of_classes = 100)

# It is recommended to specify also the final names used in the output.
tse <- estimateDiversity(tse,
  index = c("shannon", "gini_simpson", "inverse_simpson", "coverage",
               "fisher", "faith", "log_modulo_skewness"),
   name = c("Shannon", "GiniSimpson",  "InverseSimpson",  "Coverage",
               "Fisher", "Faith", "LogModSkewness"))

# The colData contains the indices by their new names provided by the user
colData(tse)[, name]

# Compare the indices visually
pairs(colData(tse)[, name])

# Plotting the diversities - use the selected names
library(scater)
plotColData(tse, "Shannon")
# ... by sample type
plotColData(tse, "Shannon", "SampleType")
\donttest{
# combining different plots
library(patchwork)
plot_index <- c("Shannon","GiniSimpson")
plots <- lapply(plot_index,
               plotColData,
               object = tse,
               x = "SampleType",
               colour_by = "SampleType")
plots <- lapply(plots,"+",
   theme(axis.text.x = element_text(angle=45,hjust=1)))
names(plots) <- plot_index
plots$Shannon + plots$GiniSimpson + plot_layout(guides = "collect")
}
}
\references{
Beisel J-N. et al. (2003)
A Comparative Analysis of Diversity Index Sensitivity.
\emph{Internal Rev. Hydrobiol.} 88(1):3-15.
\url{https://portais.ufg.br/up/202/o/2003-comparative_evennes_index.pdf}

Bulla L. (1994)
An  index of diversity and its associated diversity measure.
\emph{Oikos} 70:167--171

Faith D.P. (1992)
Conservation evaluation and phylogenetic diversity.
\emph{Biological Conservation} 61(1):1-10.

Fisher R.A., Corbet, A.S. & Williams, C.B. (1943)
The relation between the number of species and the number of individuals in
a random sample of animal population.
\emph{Journal of Animal Ecology} \emph{12}, 42-58.

Locey K.J. & Lennon J.T. (2016)
Scaling laws predict global microbial diversity.
\emph{PNAS} 113(21):5970-5975.

Magurran A.E., McGill BJ, eds (2011)
Biological Diversity: Frontiers in Measurement and Assessment.
(Oxford Univ Press, Oxford), Vol 12.

Smith B. & Wilson JB. (1996)
A Consumer's Guide to Diversity Indices.
\emph{Oikos} 76(1):70-82.
}
\seealso{
\code{\link[scater:plotColData]{plotColData}}
\itemize{
\item{\code{\link[mia:estimateRichness]{estimateRichness}}}
\item{\code{\link[mia:estimateEvenness]{estimateEvenness}}}
\item{\code{\link[mia:estimateDominance]{estimateDominance}}}
\item{\code{\link[vegan:diversity]{diversity}}}
\item{\code{\link[vegan:specpool]{estimateR}}}
}
}
\author{
Leo Lahti and Tuomas Borman. Contact: \url{microbiome.github.io}
}
