#' Plotting genes
#'
#' `geom_genes_tx` is a ggplot2 layer that visualises the positions of genes across a given region.
#' * cannot be plotted by itself, must be added to an existing ggplot object - see examples.
#'
#' @param genes.df A data.frame of genes as outputted from `get_biomart_genes`.
#' @param txdb A TxDb object as from a TxDb package.
#' @param gene_limits Set the height of the transcripts generated by `ggbio::autoplot()`. Default is NULL.
#' * If the gene is disproportionately large for the plot space, we recommend reducing the size with gene_limits = c(0,2).
#' * If there are a large amount of transcripts present, we recommend increasing the overall limit, example: gene_limits = c(0,11).
#' @param plot.space Specify gap to next plot. Recommend leaving to the default: 0.1.
#' @param plot.height Specify overall height of plot. Recommend leaving to the default: 0.3.
#'
#' @return A `ggplot_add` object.
#' @export
#' @references ggcoverage - Visualise and annotate omics coverage with ggplot2. https://github.com/showteeth/ggcoverage/tree/main
#' Yin T, Cook D, Lawrence M (2012). “ggbio: an R package for extending the grammar of graphics for genomic data.” Genome Biology, 13(8), R77.
#' @seealso [geom_peak()] [geom_dm()] [geom_counts()] [geom_gatc()] [plotWrap()] [ggplot2::ggplot_add()]
#' @examples
#' library(TxDb.Dmelanogaster.UCSC.dm6.ensGene)
#' library(org.Dm.eg.db)
#' set.seed(123)
#' example_regions <- random_regions()
#' counts.df <- random_counts()
#'
#' txdb <- TxDb.Dmelanogaster.UCSC.dm6.ensGene
#' genes <- collateGenes(txdb, example_regions, org.Dm.eg.db)
#'
#' plotCounts(counts.df,
#'     seqnames = "chr2L",
#'     start_region = 1,
#'     end_region = 40000,
#'     log2_scale = FALSE
#' ) +
#'     geom_genes_tx(genes, txdb)
#'
geom_genes_tx <- function(
    genes.df, txdb, gene_limits=NULL,
    plot.space=0.1, plot.height=0.3) {
    structure(
        list(
            genes.df = genes.df, txdb = txdb, gene_limits = gene_limits,
            plot.space = plot.space, plot.height = plot.height
        ),
        class = "genes_tx"
    )
}


#' @export
ggplot_add.genes_tx <- function(object, plot, object_name) {
    if (!is.data.frame(object$genes.df) && !inherits(object$genes.df, "GRanges")) {
        stop("data.frame/GRanges of genes is required")
    }
    if (is.null(object$gene_limits)) {
        message("If gene is disproportional to the plot, use gene_limits = c(y1,y2). If gene is too large, recommend setting to c(0,2) and adjusting the plot.height accordingly.")
    }
    plot2 <- plot
    while (inherits(plot2, "patchwork")) {
        plot2 <- plot2[[1]]
    }
    plot.data <- plot2$labels$title
    plot.data <- stringr::str_split_1(plot.data, ":")
    plot.chr <- plot.data[1]
    plot.data <- stringr::str_split_1(plot.data[2], "-")
    plot.region.start <- plot.data[1] %>% as.numeric()
    plot.region.end <- plot.data[2] %>% as.numeric()

    # get parameters
    df_og <- object$genes.df %>% data.frame()
    txdb <- object$txdb
    gene_limits <- object$gene_limits
    plot.space <- object$plot.space
    plot.height <- object$plot.height

    df <- ..getRegionsPlot(
        df = df_og, columns = c("seqnames", "start", "end", "strand", "ensembl_gene_id"),
        chr = plot.chr, start = plot.region.start,
        end = plot.region.end
    )

    if (nrow(df) == 0) {
        message("No gene data available for this region")
        gene_plot <- ggplot2::ggplot() +
            ggplot2::geom_blank()
    } else {
        df_og <- df_og %>% dplyr::filter(.data$ensembl_gene_id %in% df$ensembl_gene_id)

        exons <- ..exonCheck(txdb, plot.chr, plot.region.start, plot.region.end)

        if (nrow(exons) == 0) {
            gene_plot <- ggbio::autoplot(txdb, which = plyranges::as_granges(df_og))@ggplot
        } else {
            gene_plot <- ggbio::autoplot(txdb, which = plyranges::as_granges(df))@ggplot
        }
    }
    gene_plot <- gene_plot +
        ..themeGenePlot(
            margin.len = plot.space, plot.start = plot.region.start,
            plot.end = plot.region.end, gene_lim = gene_limits
        )

    # assemble plot
    patchwork::wrap_plots(plot + ggplot2::theme(plot.margin = ggplot2::margin(t = plot.space, b = plot.space)),
        gene_plot,
        ncol = 1, heights = c(1, plot.height)
    )
}

..exonCheck <- function(txdb, plot.chr, plot.region.start, plot.region.end) {
    df <- GenomicFeatures::exons(txdb) %>%
        data.frame() %>%
        dplyr::filter(
            .data$seqnames == plot.chr,
            .data$start >= plot.region.start,
            .data$end <= plot.region.end
        )
}

..themeGenePlot <- function(margin.len, plot.start, plot.end, gene_lim) {
    list(
        ggplot2::scale_x_continuous(expand = c(0, 0)),
        ggplot2::coord_cartesian(xlim = c(plot.start, plot.end)),
        ggplot2::labs(y = "Gene"),
        ggplot2::scale_y_continuous(
            limits = gene_lim,
            position = "right"
        ),
        ggplot2::theme_classic(),
        ggplot2::theme(
            axis.line.y = ggplot2::element_blank(),
            axis.text.y = ggplot2::element_blank(),
            axis.title.y.right = ggplot2::element_text(
                color = "black",
                angle = 90, vjust = 0.5
            ),
            axis.ticks.y = ggplot2::element_blank(),
            axis.text.x = ggplot2::element_blank(),
            axis.title.x = ggplot2::element_blank(),
            axis.ticks.x = ggplot2::element_blank(),
            panel.border = ggplot2::element_rect(
                colour = "black",
                fill = NA, linewidth = 1
            ),
            plot.margin = ggplot2::margin(t = margin.len, b = margin.len)
        )
    )
}
