% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/alluvial_plot.R
\name{alluvial_plot}
\alias{alluvial_plot}
\title{Generate an Alluvial Plot for Microbiome Data}
\source{
Built using ggalluvial, ggplot2, and dplyr for visualization of microbial abundance dynamics.
}
\usage{
alluvial_plot(
  data,
  axes = NULL,
  abundance_threshold = 10000,
  fill_variable = "Phylum",
  silent = TRUE,
  abundance_type = "absolute",
  total_reads = NULL,
  top_taxa = NULL,
  facet_vars = NULL,
  text_size = 4,
  legend_ncol = 1,
  custom_colors = color_palette$MG,
  color_mapping = NULL
)
}
\arguments{
\item{data}{A data frame containing abundance and categorical variables.}

\item{axes}{A character vector specifying the categorical variables for the x-axis.}

\item{abundance_threshold}{A numeric value specifying the minimum abundance
required for an entity to be included in the plot. Default is \code{10000}.}

\item{fill_variable}{A string specifying the variable to be used for fill colors. Default is \code{"Phylum"}.}

\item{silent}{Logical. If \code{TRUE}, suppresses warnings. Default is \code{TRUE}.}

\item{abundance_type}{A string specifying the type of abundance: \code{"absolute"} or \code{"relative"}.}

\item{total_reads}{Numeric, total number of reads for relative abundance calculation. Default is \code{NULL}.}

\item{top_taxa}{Integer. The number of top abundant taxa to retain. Default is \code{NULL}.}

\item{facet_vars}{A character vector specifying variables to facet by. Default is \code{NULL}.}

\item{text_size}{Numeric, size of text labels. Default is \code{4}.}

\item{legend_ncol}{Integer, number of columns for the legend. Default is \code{1}.}

\item{custom_colors}{A named vector specifying colors for taxa. Default is \code{color_palette$MG}.}

\item{color_mapping}{A named vector of colors for taxa, overriding \code{custom_colors}. Default is \code{NULL}.}
}
\value{
A \code{ggplot2} object representing an alluvial plot.
}
\description{
This function creates an alluvial plot based on input data, which can be
either absolute or relative abundance data.
}
\note{
This function assumes data has already been converted to long format with an "Abundance" column.
}
\examples{
if (requireNamespace("DspikeIn", quietly = TRUE) &&
    requireNamespace("phyloseq", quietly = TRUE)) {
  data("physeq_16SOTU", package = "DspikeIn")
  physeq_subset <- phyloseq::subset_samples(physeq_16SOTU, Animal.type == "Frog")
  physeq_subset <- phyloseq::prune_taxa(
    phyloseq::taxa_sums(physeq_subset) > 0, physeq_subset
  )

  # Convert phyloseq object to long format
  pps_Abs <- get_long_format_data(physeq_subset)

  # Calculate total reads (illustrative)
  total_reads <- sum(pps_Abs$Abundance)
  message("Total reads in Frog subset: ", total_reads)

  # Heavy plotting step – wrapped in \donttest{} to reduce build time
  \donttest{
    alluvial_plot_rel <- alluvial_plot(
      data = pps_Abs,
      axes = c("Env.broad.scale", "Host.genus", "Diet"),
      abundance_threshold = 0.01,
      fill_variable = "Phylum",
      abundance_type = "relative",
      top_taxa = 5,
      silent = TRUE,
      text_size = 3,
      legend_ncol = 1,
      custom_colors = DspikeIn::color_palette$cool_MG
    )
    print(alluvial_plot_rel)
  }

  # Convert to TreeSummarizedExperiment (TSE) format
  tse_data <- convert_phyloseq_to_tse(physeq_subset)
  tse_long <- get_long_format_data(tse_data)

  # Heavy plotting step – wrapped in \donttest{} to reduce build time
  \donttest{
    alluvial_plot_abs <- alluvial_plot(
      data = tse_long,
      axes = c("Env.broad.scale", "Host.genus", "Diet"),
      abundance_threshold = 2000,
      fill_variable = "Phylum",
      abundance_type = "absolute",
      top_taxa = 5,
      silent = TRUE,
      text_size = 3,
      legend_ncol = 1,
      custom_colors = DspikeIn::color_palette$cool_MG
    )
    print(alluvial_plot_abs)
  }
}
}
