#' Normalize clonal mutation counts
#'
#' This function normalizes the clonal mutation counts obtained with
#' \code{\link{clonalMutationCounter}}. The normalized counts correspond to the
#' number of mutations accumulated between conception/gastrulation and
#' MRCA/copy number gain. They can hence be interpreted as "molecular time".
#' @param countObj clonal SNV counts stratified by copy number as generated by
#' \code{\link{clonalMutationCounter}}.
#' @return a data table reporting the normalized mutation counts and densities
#' per segment, stratified by copy number
#' @examples
#' snvs <- system.file("extdata", "NBE15",
#'     "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf",
#'     package = "LACHESIS"
#' )
#' s_data <- readVCF(vcf = snvs, vcf.source = "dkfz")
#' aceseq_cn <- system.file("extdata", "NBE15",
#'     "NBE15_comb_pro_extra2.51_1.txt",
#'     package = "LACHESIS"
#' )
#' c_data <- readCNV(aceseq_cn)
#' nb <- nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)
#' cl_muts <- clonalMutationCounter(nb)
#' norm_muts <- normalizeCounts(cl_muts)
#' @import data.table
#' @export

normalizeCounts <- function(countObj = NULL) {
    n_mut_total_clonal <- A <- n_mut_A <- B <- n_mut_B <- density_total_mean <-
        Seglength <- density_A_mean <- density_B_mean <- density_total_lower <-
        density_total_upper <- density_A_lower <- density_A_upper <-
        density_B_lower <- density_B_upper <- NULL
    if (is.null(countObj)) {
        stop(
            "Please provide an countObj, as generated by clonalMutationCounter"
        )
    }

    normObj <- data.table::copy(countObj)
    # normalized mutation counts at MRCA per copy and segment
    normObj[, n_mut_total_clonal := (A * n_mut_A + B * n_mut_B +
        n_mut_total_clonal - n_mut_A -
        n_mut_B) / (A + B)]
    normObj[A == 1, n_mut_A := n_mut_total_clonal]
    normObj[B == 1, n_mut_B := n_mut_total_clonal]

    # normalized mutation densities (per megabase)
    normObj[, density_total_mean := n_mut_total_clonal / Seglength * 10^6]
    normObj[, density_A_mean := n_mut_A / Seglength * 10^6]
    normObj[, density_B_mean := n_mut_B / Seglength * 10^6]

    # 95% confidence interval using chi-square approximation for Poisson
    # distributed variables
    normObj[, density_total_lower := 0.5 *
        qchisq(0.025, n_mut_total_clonal * 2) / Seglength * 10^6]
    normObj[, density_total_upper := 0.5 *
        qchisq(0.975, n_mut_total_clonal * 2 + 2) / Seglength * 10^6]
    normObj[, density_A_lower := 0.5 * qchisq(0.025, n_mut_A * 2) /
        Seglength * 10^6]
    normObj[, density_A_upper := 0.5 * qchisq(0.975, n_mut_A * 2 + 2) /
        Seglength * 10^6]
    normObj[, density_B_lower := 0.5 * qchisq(0.025, n_mut_B * 2) /
        Seglength * 10^6]
    normObj[, density_B_upper := 0.5 * qchisq(0.975, n_mut_B * 2 + 2) /
        Seglength * 10^6]

    attr(normObj, "Raw data") <- countObj

    return(normObj)
}
