parse_s3_path <- function(path) {
  s3_provider <- .determine_s3_provider(path)

  if (is.null(s3_provider)) {
    parsed_url <- NULL
  } else if (s3_provider == "aws") {
    parsed_url <- .url_parse_aws(path)
  } else {
    parsed_url <- .url_parse_other(path)
  }

  return(parsed_url)
}

.determine_s3_provider <- function(path) {
  if (!grepl(pattern = "(^https?://)|(^s3://)", x = path)) {
    return(NULL)
  }

  matches <- regmatches(
    x = path,
    m = regexpr(
      pattern = "(amazonaws\\.com)|(embl\\.de)",
      text = path
    )
  )
  if (!length(matches)) {
    matches <- "other"
  }
  provider <- switch(
    matches,
    "amazonaws.com" = "aws",
    "embl.de" = "other",
    "other"
  )
  return(provider)
}

#' @keywords internal
.url_parse_aws <- function(url) {
  tmp <- curl::curl_parse_url(url)

  if (grepl(pattern = "^https?://s3\\.", x = url, ignore.case = TRUE)) {
    ## path style address
    bucket <- gsub(
      x = tmp$path,
      pattern = "^/?([a-z0-9\\.-]*)/.*",
      replacement = "\\1",
      ignore.case = TRUE
    )
    object <- gsub(
      x = tmp$path,
      pattern = "^/?([a-z0-9\\.-]*)/(.*)",
      replacement = "\\2",
      ignore.case = TRUE
    )
    region <- gsub(
      x = url,
      pattern = "^https?://s3\\.([a-z0-9-]*)\\.amazonaws\\.com/.*$",
      replacement = "\\1"
    )
  } else if (
    grepl(
      pattern = "^https?://[a-z0-9\\.-]*.s3\\.",
      x = url,
      ignore.case = TRUE
    )
  ) {
    ## virtual-host style address
    bucket <- gsub(
      x = tmp$host,
      pattern = "^([a-z0-9\\.-]*)\\.s3.*",
      replacement = "\\1",
      ignore.case = TRUE
    )
    object <- gsub("^/?(.*)", "\\1", tmp$path)
    region <- gsub(
      x = tmp$host,
      pattern = "^.*\\.s3\\.([a-z0-9-]*)\\.amazonaws\\.com$",
      replacement = "\\1",
      ignore.case = TRUE
    )
  } else {
    stop(
      "Unknown AWS path style.  Please report this to the package maintainer."
    )
  }

  res <- list(
    bucket = bucket,
    object = object,
    region = region,
    hostname = "https://s3.amazonaws.com"
  )
  return(res)
}

#' @keywords internal
.url_parse_other <- function(url) {
  parsed_url <- curl::curl_parse_url(url)
  bucket <- gsub(
    x = parsed_url$path,
    pattern = "^/?([[a-z0-9:\\.-]*)/.*",
    replacement = "\\1",
    ignore.case = TRUE
  )
  object <- gsub(
    x = parsed_url$path,
    pattern = "^/?([a-z0-9:\\.-]*)/(.*)",
    replacement = "\\2",
    ignore.case = TRUE
  )
  hostname <- paste0(parsed_url$scheme, "://", parsed_url$host)

  if (!is.null(parsed_url$port)) {
    hostname <- paste0(hostname, ":", parsed_url$port)
  }

  res <- list(
    bucket = bucket,
    object = object,
    region = "auto",
    hostname = hostname
  )
  return(res)
}

#' This is a modified version of paws.storage:::get_credentials().  It is
#' included to prevent using the `:::` operator.  Look at that function if
#' things stop working.
#'
#' @param credentials Content stored at `.internal$config$credentials` in
#' an object created by `paws.storage::s3()`.
#'
#' @returns A credentials list to be reinserted into a `paws.storage` s3 object.
#' If no valid credentials are found this function will error, which is expected
#' and is caught by `.check_credentials`.
#'
#' @importFrom methods formalArgs
#' @keywords internal
.get_credentials <- function(credentials) {
  for (provider in credentials$provider) {
    args <- formalArgs(provider)
    if (is.null(args)) {
      creds <- provider()
    } else {
      creds <- do.call(provider, as.list(credentials)[args])
    }
    if (!is.null(creds)) {
      credentials$creds <- creds
      break
    }
  }
  return(credentials)
}

.check_credentials <- function(s3_client, parsed_url) {
  test <- try(
    .get_credentials(s3_client$.internal$config$credentials),
    silent = TRUE
  )

  if (inherits(test, "try-error")) {
    s3_client <- s3(
      config = list(
        credentials = list(
          anonymous = TRUE
        ),
        region = parsed_url$region,
        endpoint = parsed_url$hostname
      )
    )
  }

  return(s3_client)
}


#' @importFrom paws.storage s3
.create_s3_client <- function(path) {
  parsed_url <- parse_s3_path(path)

  if (is.null(parsed_url)) {
    s3_client <- NULL
  } else {
    s3_client <- s3(
      config = list(
        region = parsed_url$region,
        endpoint = parsed_url$hostname
      )
    )

    s3_client <- .check_credentials(s3_client, parsed_url)
  }
  return(s3_client)
}

.s3_object_exists <- function(s3_client, Bucket, Key) {
  exists <- s3_client$list_objects_v2(
    Bucket = Bucket,
    Prefix = Key
  )$KeyCount > 0

  return(exists)
}
