% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Spectra.R
\name{combinePeaks}
\alias{combinePeaks}
\alias{combinePeaks,Spectra-method}
\title{Aggregating and combining mass peaks data}
\usage{
\S4method{combinePeaks}{Spectra}(
  object,
  tolerance = 0,
  ppm = 20,
  intensityFun = base::mean,
  mzFun = base::mean,
  weighted = TRUE,
  msLevel. = uniqueMsLevels(object),
  ...
)
}
\arguments{
\item{object}{A \code{Spectra} object.}

\item{tolerance}{\code{numeric(1)} allowing to define a constant maximal
accepted difference between m/z values for peaks to be grouped. Default
is \code{tolerance = 0}.}

\item{ppm}{\code{numeric(1)} defining a relative, m/z-dependent, maximal
accepted difference between m/z values for peaks to be grouped. Default
is \code{ppm = 20}.}

\item{intensityFun}{Function to aggregate intensities for all peaks in
each peak group into a single intensity value.}

\item{mzFun}{Function to aggregate m/z values for all mass peaks within
each peak group into a single m/z value. This parameter is ignored if
\code{weighted = TRUE} (the default).}

\item{weighted}{\code{logical(1)} whether m/z values of peaks within each peak
group should be aggregated into a single m/z value using an
intensity-weighted mean. Defaults to \code{weighted = TRUE}.}

\item{msLevel.}{\code{integer} defining the MS level(s) of the spectra to which
the function should be applied (defaults to all MS levels of \code{object}.}

\item{...}{ignored.}
}
\description{
In addition to aggregating content of spectra variables (describe in
\code{\link[=combineSpectra]{combineSpectra()}}) it is also possible to aggregate and combine mass peaks
data from individual spectra within a \code{Spectra}. These \code{combinePeaks()}
function combines mass peaks \strong{within each spectrum} with a difference in
their m/z values that is smaller than the maximal acceptable difference
defined by \code{ppm} and \code{tolerance}. Parameters \code{intensityFun} and \code{mzFun}
allow to define functions to aggregate the intensity and m/z values for
each such group of peaks. With \code{weighted = TRUE} (the default), the m/z
value of the combined peak is calculated using an intensity-weighted mean
and parameter \code{mzFun} is ignored. The \code{\link[MsCoreUtils:group]{MsCoreUtils::group()}} function is
used for the grouping of mass peaks. Parameter \code{msLevel.} allows to define
selected MS levels for which peaks should be combined. This function
returns a \code{Spectra} with the same number of spectra than the input object,
but with possibly combined peaks within each spectrum.
Additional peak variables (other than \code{"mz"} and \code{"intensity"}) are
dropped (i.e. their values are replaced with \code{NA}) for combined peaks
unless they are constant across the combined peaks. See also
\code{\link[=reduceSpectra]{reduceSpectra()}} for a function to select a single \emph{representative}
mass peak for each peak group.
}
\examples{

## Create a Spectra from mzML files and use the `MsBackendMzR` on-disk
## backend.
sciex_file <- dir(system.file("sciex", package = "msdata"),
    full.names = TRUE)
sciex <- Spectra(sciex_file, backend = MsBackendMzR())

## Combine mass peaks per spectrum with a difference in their m/z value
## that is smaller than 20 ppm. The intensity values of such peaks are
## combined by summing their values, while for the m/z values the median
## is reported
sciex_comb <- combinePeaks(sciex, ppm = 20,
    intensityFun = sum, mzFun = median)

## Comparing the number of mass peaks before and after aggregation
lengths(sciex) |> head()
lengths(sciex_comb) |> head()

## Plotting the first spectrum before and after aggregation
par(mfrow = c(1, 2))
plotSpectra(sciex[2L])
plotSpectra(sciex_comb[2L])

## Using `reduceSpectra()` to keep for each group of mass peaks with a
## difference in their m/z values < 20ppm the one with the highest intensity.
sciex_red <- reduceSpectra(sciex, ppm = 20)

## Comparing the number of mass peaks before and after the operation
lengths(sciex) |> head()
lengths(sciex_red) |> head()
}
\seealso{
\itemize{
\item \code{\link[=combineSpectra]{combineSpectra()}} for functions to combine or aggregate \code{Spectra}'s
spectra data.
\item \code{\link[=combinePeaksData]{combinePeaksData()}} for the function to combine the mass peaks data.
\item \code{\link[=reduceSpectra]{reduceSpectra()}} and similar functions to filter mass peaks data.
\item \link{Spectra} for a general description of the \code{Spectra} object.
}
}
\author{
Sebastian Gibb, Johannes Rainer, Laurent Gatto
}
