% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mapCellsToEdges.R
\name{mapCellsToEdges}
\alias{mapCellsToEdges}
\alias{mapCellsToEdges,ANY-method}
\alias{mapCellsToEdges,SummarizedExperiment-method}
\alias{mapCellsToEdges,SingleCellExperiment-method}
\title{Map cells to edges}
\usage{
mapCellsToEdges(x, ...)

\S4method{mapCellsToEdges}{ANY}(x, mst, clusters, columns = NULL)

\S4method{mapCellsToEdges}{SummarizedExperiment}(x, ..., assay.type = "logcounts")

\S4method{mapCellsToEdges}{SingleCellExperiment}(
  x,
  clusters = colLabels(x, onAbsence = "error"),
  ...,
  use.dimred = NULL
)
}
\arguments{
\item{x}{A numeric matrix of coordinates where each row represents a cell/sample and each column represents a dimension
(usually a PC or another low-dimensional embedding, but features or genes can also be used).

Alternatively, a \linkS4class{SummarizedExperiment} or \linkS4class{SingleCellExperiment} object
containing such a matrix in its \code{\link{assays}}, as specified by \code{assay.type}.
This will be transposed prior to use.

Alternatively, for \linkS4class{SingleCellExperiment}s, this matrix may be extracted from its \code{\link{reducedDims}},
based on the \code{use.dimred} specification.
In this case, no transposition is performed.}

\item{...}{For the generic, further arguments to pass to the specific methods.

For the SummarizedExperiment method, further arguments to pass to the ANY method.

For the SingleCellExperiment method, further arguments to pass to the SummarizedExperiment method
(if \code{use.dimred} is specified) or the ANY method (otherwise).}

\item{mst}{A \link{graph} object containing a MST,
constructed from the same coordinate space as the values in \code{x} (e.g., same PC space, same set of features).}

\item{clusters}{A factor-like object of the same length as \code{nrow(x)},
specifying the cluster identity for each cell in \code{x}.
This can also be \code{NULL}, see details.}

\item{columns}{A character, logical or integer vector specifying the columns of \code{x} to use.
If \code{NULL}, all provided columns are used by default.}

\item{assay.type}{An integer or string specifying the assay to use from a SummarizedExperiment \code{x}.}

\item{use.dimred}{An integer or string specifying the reduced dimensions to use from a SingleCellExperiment \code{x}.}
}
\value{
A \linkS4class{DataFrame} with one row per row of \code{x}, containing the fields:
\itemize{
\item \code{left.cluster}, the cluster on one end of the edge to which the cell was assigned.
\item \code{right.cluster}, the cluster on the other end of the edge to which the cell was assigned.
\item \code{left.distance}, the distance to the cluster centroid on one end.
\item \code{right.distance}, the distance to the cluster centroid on the other end.
}
Note that the sum of the distances will always yield the edge length.
}
\description{
Map each cell to the closest edge on the MST, reporting also the distance to the corresponding vertices.
}
\details{
For each cluster, we consider all edges of the MST involving that cluster.
Each cell of that cluster is then mapped to the closest of these edges (where proximity is defined by Euclidean distance).
The identity of and distance from each ends of the edge is reported;
this can be useful for downstream pseudo-time calculations or to subset cells by those lying on a particular edge.

If \code{clusters=NULL}, each cell can be mapped to \emph{any} edge of the MST.
This is useful if the \code{mst} was constructed from a different set of cells than those in \code{x},
allowing us to effectively project new datasets onto an existing MST.
Note, however, that the new \code{x} must lie in the same coordinate space as the \code{x} used to make \code{mst}.

Some cells may simply be mapped to the edge endpoints.
This manifests as values of zero for the distances from either end of the edge.
For analyses focusing on a specific edge, it may be advisable to filter out such cells 
as their edge assignments are arbitrarily assigned and they do not contribute to any transitional process along the edge.
}
\examples{
# Mocking up a Y-shaped trajectory.
centers <- rbind(c(0,0), c(0, -1), c(1, 1), c(-1, 1))
rownames(centers) <- seq_len(nrow(centers))
clusters <- sample(nrow(centers), 1000, replace=TRUE)
cells <- centers[clusters,]
cells <- cells + rnorm(length(cells), sd=0.5)

# Creating the MST first:
mst <- createClusterMST(cells, clusters=clusters)
plot(mst)

# Mapping cells to the MST:
mapping <- mapCellsToEdges(cells, mst, clusters=clusters)
head(mapping)

# Also works with some random extra cells:
extras <- matrix(rnorm(1000), ncol=2)
emapping <- mapCellsToEdges(extras, mst, clusters=NULL)
head(emapping)

}
\references{
Ji Z and Ji H (2016).
TSCAN: Pseudo-time reconstruction and evaluation in single-cell RNA-seq analysis.
\emph{Nucleic Acids Res.} 44, e117
}
\seealso{
\code{\link{createClusterMST}}, to generate \code{mst}.

\code{\link{quickPseudotime}}, a wrapper to quickly perform these calculations.
}
\author{
Aaron Lun
}
