#' Smooth signature scores by kNN
#'
#' This function performs smoothing of single-cell scores by weighted
#' average of the k-nearest neighbors. It can be useful to 'impute' scores by
#' neighboring cells and partially correct data sparsity. While this function
#' has been designed to smooth UCell scores, it can be applied to any numerical
#' metadata contained in SingleCellExperiment or Seurat objects
#'
#' @param obj Input object - either a [SingleCellExperiment] object
#'     or a Seurat object.
#' @param signature.names The names of the signatures (or any numeric metadata
#'     column) for which to calculate kNN-smoothed scores
#' @param reduction Which dimensionality reduction to use for kNN smoothing.
#'     It must be already present in the input object.
#' @param k Number of neighbors for kNN smoothing
#' @param decay Exponential decay for nearest neighbor weight: (1-decay)^n
#' @param up.only If set to TRUE, smoothed scores will only be allowed
#'     to increase by smoothing
#' @param BNPARAM A [BiocNeighborParam] object specifying the algorithm to use
#'     for kNN calculation.
#' @param BPPARAM A [BiocParallel::bpparam()] object for parallel computing,
#'     e.g. [MulticoreParam] or [SnowParam]
#' @param suffix Suffix to append to metadata columns
#'     for the new knn-smoothed scores  
#' @param assay For Seurat objects only - do smoothing on expression
#'     data from this assay. When NULL, only looks in metadata
#' @param slot For Seurat objects only - do smoothing on expression
#'     data from this slot
#' @param sce.expname For sce objects only - which experiment stores the
#'    signatures to be smoothed. Set to 'main' for smoothing gene expression
#'    stored in the main sce experiment.
#' @param sce.assay For sce objects only - pull data from this assay
#' @return An augmented \code{obj} with the smoothed signatures. If \code{obj}
#'    is a Seurat object, smoothed signatures are added to metadata; if 
#'    \code{obj} is a SingleCellExperiment object, smoothed signatures are 
#'    returned in a new altExp. See the examples below.     
#' @examples
#' #### Using Seurat ####
#' library(Seurat)
#' gene.sets <- list(Tcell = c("CD2","CD3E","CD3D"),
#'                 Myeloid = c("SPI1","FCER1G","CSF1R"))
#' data(sample.matrix)
#' obj <- Seurat::CreateSeuratObject(sample.matrix)                
#' # Calculate UCell scores
#' obj <- AddModuleScore_UCell(obj,features = gene.sets, name=NULL)
#' # Run PCA
#' obj <- FindVariableFeatures(obj) |> NormalizeData() |> ScaleData() |> RunPCA(npcs=5)
#' # Smooth signatures
#' obj <- SmoothKNN(obj, k=3, signature.names=names(gene.sets))
#' head(obj[[]])
#' 
#' #### Using SingleCellExperiment ####
#' library(SingleCellExperiment)
#' library(scater)
#' data(sample.matrix)
#' sce <- SingleCellExperiment(list(counts=sample.matrix))
#' gene.sets <- list( Tcell = c("CD2","CD3E","CD3D"),
#'                   Myeloid = c("SPI1","FCER1G","CSF1R"))
#' # Calculate UCell scores
#' sce <- ScoreSignatures_UCell(sce, features=gene.sets, name=NULL)
#' # Run PCA
#' sce <- logNormCounts(sce)
#' sce <- runPCA(sce, scale=TRUE, ncomponents=5)
#' # Smooth signatures
#' sce <- SmoothKNN(sce, k=3, signature.names=names(gene.sets))
#' # See results
#' altExp(sce, 'UCell')
#' assays(altExp(sce, 'UCell'))
#' # Plot on UMAP
#' sce <- runUMAP(sce, dimred="PCA")
#' plotUMAP(sce, colour_by = "Tcell_kNN", by_exprs_values = "UCell_kNN")
#' 
#' @importFrom methods setMethod setGeneric is
#' @import BiocNeighbors
#' @importFrom Matrix t
#' @export SmoothKNN
SmoothKNN <- function(
    obj=NULL,
    signature.names=NULL,
    reduction="pca",
    k=10,
    decay=0.1,
    up.only=FALSE,
    BNPARAM=AnnoyParam(),
    BPPARAM=SerialParam(),
    suffix="_kNN",
    assay=NULL,
    slot="data",
    sce.expname=c("UCell","main"),
    sce.assay=NULL)
{
    UseMethod("SmoothKNN")
}

