#' Plot an UpSet Diagram from Genomic or Set Overlap Results
#'
#' This function creates an UpSet plot using the `ComplexHeatmap` package to
#' visualize intersections across multiple sets.
#' Supports both `GenomicOverlapsResult` and `SetOverlapsResult` objects.
#'
#' @param overlap_object A `GenomicOverlapsResult` or `SetOverlapsResult` object
#' returned by \code{\link{computeOverlaps}}.
#' @param customSetOrder Optional. A vector specifying the order of sets to
#'   display on the UpSet diagram. The vector should contain either numeric
#'   indices (corresponding to the sets in the overlap object) or character
#'   names (matching the set names). If `NULL` (default), sets are displayed
#'   in decreasing order of their size (\code{set_size()}).
#' @param comb_col Optional. Color(s) for the combination matrix dots and
#'   connecting lines. Can be a single color, a vector of colors (recycled to
#'   match the number of intersections). Default is "black".
#'
#' @return An UpSet plot object generated by \code{ComplexHeatmap::UpSet}.
#' @export
#'
#' @examples
#' # Example with gene sets (built-in dataset)
#' data(gene_list)
#' res_sets <- computeOverlaps(gene_list)
#'
#' # Default order (sets sorted by size)
#' plotUpSet(res_sets)
#'
#' # Custom color
#' plotUpSet(res_sets, comb_col = "darkblue")
#'
#' # Custom order by names
#' plotUpSet(res_sets, customSetOrder = c("random_genes_C",
#'                                        "random_genes_A",
#'                                        "random_genes_B"))
#'
#' # Example with genomic regions (built-in dataset)
#' data(a549_chipseq_peaks)
#' res_genomic <- computeOverlaps(a549_chipseq_peaks)
#' plotUpSet(res_genomic)
plotUpSet <- function(overlap_object, customSetOrder = NULL, comb_col = "black") {
    if (!inherits(overlap_object, "GenomicOverlapResult") &&
        !inherits(overlap_object, "SetOverlapResult")) {
        stop("Input must be a GenomicOverlapResult or SetOverlapResult.")
    }

    overlap_matrix <- overlap_object[["overlap_matrix"]]
    combMat <- ComplexHeatmap::make_comb_mat(overlap_matrix)

    annot_top <- ComplexHeatmap::HeatmapAnnotation(
        "Intersection\nsize" = ComplexHeatmap::anno_barplot(
            ComplexHeatmap::comb_size(combMat),
            border = FALSE,
            gp = grid::gpar(fill = "black"),
            height = grid::unit(2, "cm")
        ),
        "Size" = ComplexHeatmap::anno_text(
            ComplexHeatmap::comb_size(combMat),
            rot = 0,
            just = "center",
            location = 0.25
        ),
        annotation_name_side = "left",
        annotation_name_rot = 0
    )

    annot_right <- ComplexHeatmap::rowAnnotation(
        "Set size" = ComplexHeatmap::anno_barplot(
            ComplexHeatmap::set_size(combMat),
            border = FALSE,
            gp = grid::gpar(fill = "black"),
            width = grid::unit(2, "cm")
        ),
        "Size" = ComplexHeatmap::anno_text(ComplexHeatmap::set_size(combMat))
    )

    # Default: order sets by decreasing size
    if (is.null(customSetOrder)) {
        customSetOrder <- order(ComplexHeatmap::set_size(combMat),
                                decreasing = TRUE)
    }

    upset <- ComplexHeatmap::UpSet(
        combMat,
        top_annotation = annot_top,
        right_annotation = annot_right,
        set_order = customSetOrder,
        comb_col = comb_col
    )
    return(upset)
}
