#' iSEE layout for TreeSE
#' 
#' Panel configuration tuned to the specific properties of
#' \code{\link[TreeSummarizedExperiment:TreeSummarizedExperiment-constructor]{TreeSummarizedExperiment}}.
#' 
#' @inherit iSEE::iSEE params details sections
#' 
#' @return
#' The iSEE method for the TreeSE container returns a default set of panels
#' typically relevant for microbiome data. This configuration can be modified
#' by defining a different set of initial panels. By default, the interface
#' includes the following panels:
#' \itemize{
#' \item \code{RowDataTable()}
#' \item \code{ColumnDataTable()}
#' \item \code{RowTreePlot()}
#' \item \code{AbundancePlot()}
#' \item \code{AbundanceDensityPlot()}
#' \item \code{ReducedDimensionPlot()}
#' \item \code{ComplexHeatmapPlot()}
#' }
#'
#' @examples
#' # Import TreeSE
#' library(mia)
#' data("GlobalPatterns", package = "mia")
#' tse <- GlobalPatterns
#'
#' # Agglomerate TreeSE by Genus
#' tse_genus <- agglomerateByRank(tse,
#'                                rank = "Genus",
#'                                onRankOnly = TRUE)
#'
#' # Add relabundance assay
#' tse_genus <- transformAssay(tse_genus, method = "relabundance")
#'
#' # Launch iSEE with custom initial panels
#' if (interactive()) {
#'   iSEE(tse_genus, initial = c(RowTreePlot(), AbundancePlot(), AbundanceDensityPlot()))
#' }
#' 
#' @docType methods
#' @aliases iSEE,TreeSummarizedExperiment-method
#' @name iSEE
NULL

#' @rdname iSEE
#' @export
setGeneric("iSEE", iSEE::iSEE)

#' @export
#' @importFrom iSEE createLandingPage ExperimentColorMap ReducedDimensionPlot
#'   ComplexHeatmapPlot RowDataTable ColumnDataTable
setMethod("iSEE", "TreeSummarizedExperiment",
    function(se, initial = c(RowDataTable(), ColumnDataTable(), RowTreePlot(),
        AbundancePlot(), AbundanceDensityPlot(), ReducedDimensionPlot(),
        ComplexHeatmapPlot()), extra = NULL, colormap = ExperimentColorMap(),
        landingPage = createLandingPage(), tour = NULL, appTitle = NULL,
        tabTitle = NULL, runLocal = TRUE, voice = FALSE, bugs = FALSE,
        saveState = NULL, ...) {
    
    initial <- .check_all_panels(se, initial)

    iSEE::iSEE(se, initial = initial, extra = initial, colormap = colormap,
        landingPage = landingPage, tour = tour, appTitle = appTitle,
        runLocal = runLocal, voice = voice, bugs = bugs,
        saveState = saveState, ...)
})