% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/spectrum.R
\docType{methods}
\name{spectrumKernel}
\alias{getFeatureSpaceDimension}
\alias{getFeatureSpaceDimension,ANY-method}
\alias{getFeatureSpaceDimension,SpectrumKernel-method}
\alias{spectrumKernel}
\title{Spectrum Kernel}
\usage{
spectrumKernel(k = 3, r = 1, annSpec = FALSE, distWeight = numeric(0),
  normalized = TRUE, exact = TRUE, ignoreLower = TRUE, presence = FALSE,
  revComplement = FALSE, mixCoef = numeric(0))

\S4method{getFeatureSpaceDimension}{SpectrumKernel}(kernel, x)
}
\arguments{
\item{k}{length of the substrings (also called kmers). This parameter defines
the size of the feature space, i.e. the total number of features considered
in this kernel is |A|^k, with |A| as the size of the alphabet (4 for DNA
and RNA sequences and 21 for amino acid sequences). When multiple kernels
with different k values should be generated e.g. for model selection a
range e.g. k=3:5 can be specified. In this case a list of kernel objects
with the individual k values from the range is generated as result.
Default=3}

\item{r}{exponent which must be > 0 (details see below). Default=1}

\item{annSpec}{boolean that indicates whether sequence annotation should
be taken into account (details see on help page for
\code{\link{annotationMetadata}}). For the annotation specific spectrum
kernel the total number of features increases to |A|^k * |a|^k with
|A| as the size of the sequence alphabet and |a| as the size of the
annotation alphabet. Default=FALSE}

\item{distWeight}{a numeric distance weight vector or a distance weighting
function (details see on help page for \code{\link{gaussWeight}}).
Default=NULL}

\item{normalized}{a kernel matrix or explicit representation generated with
this kernel will be normalized(details see below). Default=TRUE}

\item{exact}{use exact character set for the evaluation (details see below).
Default=TRUE}

\item{ignoreLower}{ignore lower case characters in the sequence. If the
parameter is not set lower case characters are treated like uppercase.
Default=TRUE}

\item{presence}{if this parameter is set only the presence of a kmers will
be considered, otherwise the number of occurances of the kmer is used.
Default=FALSE}

\item{revComplement}{if this parameter is set a kmer and its reverse
complement are treated as the same feature. Default=FALSE}

\item{mixCoef}{mixing coefficients for the mixture variant of the spectrum
kernel. A numeric vector of length k is expected for this parameter with
the unused components in the mixture set to 0. Default=numeric(0)}

\item{kernel}{a sequence kernel object}

\item{x}{one or multiple biological sequences in the form of a
\code{\linkS4class{DNAStringSet}}, \code{\linkS4class{RNAStringSet}},
\code{\linkS4class{AAStringSet}} (or as \code{\linkS4class{BioVector}})}
}
\value{
spectrumKernel: upon successful completion, the function returns a kernel
object of class \code{\linkS4class{SpectrumKernel}}.

of getDimFeatureSpace:
dimension of the feature space as numeric value
}
\description{
Create a spectrum kernel object
}
\details{
Creation of kernel object\cr\cr
The function 'spectrumKernel' creates a kernel object for the spectrum
kernel. This kernel object can then be used with a set of DNA-, RNA- or
AA-sequences to generate a kernel matrix or an explicit representation for
this kernel. The spectrum kernel uses all subsequences for length k (also
called kmers). For sequences shorter than k the self similarity (i.e. the
value on the main diagonal in the square kernel matrix) is 0. The explicit
representation contains only zeros for such a sample. Dependent on the
learning task it might make sense to remove such sequences from the data
set as they do not contribute to the model but still influence performance
values. \cr\cr
For values different from 1 (=default value) parameter \code{r}
leads to a transfomation of similarities by taking each element of the
similarity matrix to the power of r. Only integer values larger than 1
should be used for r in context with SVMs requiring positive definite
kernels. If \code{normalized=TRUE}, the feature vectors are scaled to the
unit sphere before computing the similarity value for the kernel matrix.
For two samples with the feature vectors \code{x} and \code{y} the
similarity is computed as:
\deqn{s=\frac{\vec{x}^T\vec{y}}{\|\vec{x}\|\|\vec{y}\|}}{s=(x^T y)/(|x| |y|)}
For an explicit representation generated with the feature map of a
normalized kernel the rows are normalized by dividing them through their
Euclidean norm. For parameter \code{exact=TRUE} the sequence characters
are interpreted according to an exact character set. If the flag is not
set ambigous characters from the IUPAC characterset are also evaluated.
For sequences shorter than k the self similarity (i.e. the value on the
main diagonal in the square kernel matrix) is 0.

The annotation specific variant (for details see
\link{annotationMetadata}) and the position dependent variants (for
details see \link{positionMetadata}) either in the form of a position
specific or a distance weighted kernel are supported for the spectrum
kernel. The generation of an explicit representation is not possible for
the position dependent variants of this kernel.\cr\cr
Creation of kernel matrix\cr\cr
The kernel matrix is created with the function \code{\link{getKernelMatrix}}
or via a direct call with the kernel object as shown in the examples below.
}
\examples{
## instead of user provided sequences in XStringSet format
## for this example a set of DNA sequences is created
## RNA- or AA-sequences can be used as well with the spectrum kernel
dnaseqs <- DNAStringSet(c("AGACTTAAGGGACCTGGTCACCACGCTCGGTGAGGGGGACGGGGTGT",
                          "ATAAAGGTTGCAGACATCATGTCCTTTTTGTCCCTAATTATTTCAGC",
                          "CAGGAATCAGCACAGGCAGGGGCACGGCATCCCAAGACATCTGGGCC",
                          "GGACATATACCCACCGTTACGTGTCATACAGGATAGTTCCACTGCCC",
                          "ATAAAGGTTGCAGACATCATGTCCTTTTTGTCCCTAATTATTTCAGC"))
names(dnaseqs) <- paste("S", 1:length(dnaseqs), sep="")

## create the kernel object for dimers without normalization
speck <- spectrumKernel(k=2, normalized=FALSE)
## show details of kernel object
speck

## generate the kernel matrix with the kernel object
km <- speck(dnaseqs)
dim(km)
km[1:5,1:5]

## alternative way to generate the kernel matrix
km <- getKernelMatrix(speck, dnaseqs)
km[1:5,1:5]

\dontrun{
## plot heatmap of the kernel matrix
heatmap(km, symm=TRUE)
}
}
\author{
Johannes Palme
}
\references{
\url{https://github.com/UBod/kebabs}\cr\cr
C.S. Leslie, E. Eskin and W.S. Noble (2002) The spectrum kernel:
a string kernel for SVM protein classification.
\emph{Proc. Pacific Symposium on Biocomputing}, pp. 566-575.
\cr\cr
U. Bodenhofer, K. Schwarzbauer, M. Ionescu, and
S. Hochreiter (2009)
Modelling position specificity in sequence kernels by fuzzy
equivalence relations. \emph{Proc. Joint 13th IFSA World Congress and 6th
EUSFLAT Conference}, pp. 1376-1381, Lisbon.\cr\cr
C.C. Mahrenholz, I.G. Abfalter, U. Bodenhofer, R. Volkmer and
S. Hochreiter (2011) Complex networks govern coiled coil
oligomerization - predicting and profiling by means of a machine
learning approach. \emph{Mol. Cell. Proteomics}, 10(5):M110.004994.
DOI: \doi{10.1074/mcp.M110.004994}. \cr\cr
J. Palme, S. Hochreiter, and U. Bodenhofer (2015) KeBABS: an R package
for kernel-based analysis of biological sequences.
\emph{Bioinformatics}, 31(15):2574-2576.
DOI: \doi{10.1093/bioinformatics/btv176}.
}
\seealso{
\code{\link{kernelParameters-method}},
\code{\link{getKernelMatrix}}, \code{\link{getExRep}},
\code{\link{mismatchKernel}}, \code{\link{motifKernel}},
\code{\link{gappyPairKernel}}, \code{\linkS4class{SpectrumKernel}}
}
\keyword{kernel}
\keyword{methods}
\keyword{spectrum}
\keyword{spectrumKernel}

