\name{createSTDdbGC}
\alias{createSTDdbGC}
\alias{generateStdDBGC}
\title{Create an in-house database for GC-MS annotation}
\description{
  For creating an in-house instrument-specific annotation database,
  injections of pure standards need to be processed. All patterns in the
  vicinity of the retention time of the standard (to be provided by the
  user) will be compared to an external database - in case of a sufficient
  match, they will be retained in the database. The
  \code{generateStdDBGC} is not meant to be called directly by the user.}
\usage{
createSTDdbGC(stdInfo, settings, extDB = NULL, manualDB = NULL,
              RIstandards = NULL, nSlaves = 0)
generateStdDBGC(totalXset, settings, extDB = NULL, manualDB = NULL,
                RIstandards = NULL)
}
\arguments{
  \item{stdInfo}{Information of the standards, given in the form of a
    \code{data.frame}. Minimal information: \code{stdFile}, \code{Name},
    \code{CAS}, monoisotopic mass (\code{monoMW}), and retention time
    (\code{rt}). The filenames in slot \code{stdFile} should include
    path information. If this argument is NULL, this function can be
    used to process a manually curated DB. Arguments \code{stdInfo} and
    \code{manualDB} cannot be both NULL.}
  \item{settings}{A list of settings, to be used in peak picking and
    pattern comparison.}
  \item{extDB}{The external database containing spectra, with which to
    compare the patterns found in the standards files.}
  \item{manualDB}{A database of manually curated spectra, that will be
    incorporated in the final DB without any further checks.}
  \item{totalXset}{A list of xset objects, as generated by
    \code{peakDetection}.}
  \item{RIstandards}{A two-column matrix containing for the standards
    defining the RI scale both retention times and retention indices. If
    not given, no RI values will be calculated and retention times will
    be used instead.}
  \item{nSlaves}{Number of cores to be used in peak picking.}
}
\details{Function \code{createSTDdbGC} creates a database object
  containing validated pseudospectra for a number of compounds. The
  injections of the standards, described in the input object
  \code{stdInfo}, are processed using function \code{processStandards};
  comparison with the external database, inclusion of manual compounds
  and final formatting are done in function
  \code{generateStdDBGC}. Several situations can be envisaged:

  A: a series of injections of standards needs to be compared with a
  standard library, such as the NIST. In this case, both \code{stdInfo}
  and \code{extDB} need to be non-null, and the result will be a
  database in which the entries have a sufficient match with the
  external DB. If \code{manualDB} is also non-null, these entries will
  be added too (without checking).

  B: for a series of injections no standard library information is
  available (\code{extDB} is NULL, and \code{stdInfo} is not), and the
  function simply returns all patterns eluting around 
  the indicated retention time. This allows for subsequent manual
  validation and pruning. If \code{manualDB} is non-null, these
  entries will be added, but since this is a somewhat unusual thing to
  do, a warning will be given.

  C: a manual database needs to be processed to be useable as a real
  database. This basically entails renaming the \code{rt} and
  \code{rt.sd} fields into \code{std.rt} and \code{std.rt.sd}, and
  a similar action for any \code{RI} field.
}
\value{The output of \code{createSTDdbGC} (and \code{generateStdDBGC},
  which is the last function called in \code{createSTDdbGC}) is
  a list, where every entry describes one compound/spectrum
  combination. For use in annotation, the following fields are
  mandatory: \code{Name}, \code{std.rt}, \code{pspectrum} and
  \code{monoMW}.
}
\author{Ron Wehrens}
\seealso{\code{\link{processStandards}}, \code{\link{generateStdDBGC}}}
\examples{
  data(threeStdsNIST)  ## provides object smallDB, excerpt from NIST DB
  \dontrun{
if (require(metaMSdata)) {
  ## Sitation A: create a DB of standards.
  ## first tell the system where to look
  data(threeStdsInfo)
  all.files <- list.files(system.file("extdata", package = "metaMSdata"),
                          pattern = "_GC_", full.names = TRUE)
  stdInfo[,"stdFile"] <- rep(all.files[3], 3)

  data(FEMsettings)    ## provides a.o. TSQXLS.GC, the GC settings file
  data(threeStdsNIST)  ## provides object smallDB, excerpt from NIST DB

  DB <- createSTDdbGC(stdInfo, TSQXLS.GC, extDB = smallDB)
  }
  ## saved in "threeStdsDB.RData" in the data directory of the metaMS
  ## package

  ## Situation B: do not check the data with an external database. Now
  ## the fields bestDBmatch and validation will be absent.
  DB <- createSTDdbGC(stdInfo, TSQXLS.GC, extDB = NULL)

  ## Situation C: create a DB directly from an msp file (manual DB)
  manual.fname <- list.files(system.file("extdata", package = "metaMSdata"),
                             pattern = "msp", full.names = TRUE)
  manual <- read.msp(manual.fname)
  DB <- createSTDdbGC(stdInfo = NULL, settings = TSQXLS.GC,
                      manualDB = manual)
}
}
\keyword{manip}
