% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findNhoodMarkers.R
\name{findNhoodMarkers}
\alias{findNhoodMarkers}
\title{Identify post-hoc neighbourhood marker genes}
\arguments{
\item{x}{A \code{\linkS4class{Milo}} object containing single-cell gene expression
and neighbourhoods.}

\item{da.res}{A \code{data.frame} containing DA results, as expected from running
\code{testNhoods}.}

\item{da.fdr}{A numeric scalar that determines at what FDR neighbourhoods are declared
DA for the purposes of aggregating across concorantly DA neighbourhoods.}

\item{assay}{A character scalar determining which \code{assays} slot to extract from the
\code{\linkS4class{Milo}} object to use for DGE testing.}

\item{aggregate.samples}{logical indicating wheather the expression values for cells in the same sample
and neighbourhood group should be merged for DGE testing. This allows to perform testing exploiting the replication structure
in the experimental design, rather than treating single-cells as independent replicates. The function used for aggregation depends on the
selected gene expression assay: if \code{assay="counts"} the expression values are summed, otherwise we take the mean.}

\item{sample_col}{a character scalar indicating the column in the colData storing sample information
(only relevant if \code{aggregate.samples==TRUE})}

\item{overlap}{A scalar integer that determines the number of cells that must
overlap between adjacent neighbourhoods for merging.}

\item{lfc.threshold}{A scalar that determines the absolute log fold change above
which neighbourhoods should be considerd 'DA' for merging. Default=NULL}

\item{merge.discord}{A logical scalar that overrides the default behaviour and allows
adjacent neighbourhoods to be merged if they have discordant log fold change signs. Using
this argument is generally discouraged, but may be useful for constructing an empirical null
group of cells, regardless of DA sign.}

\item{subset.row}{A logical, integer or character vector indicating the rows
of \code{x} to use for sumamrizing over cells in neighbourhoods.}

\item{gene.offset}{A logical scalar the determines whether a per-cell offset
is provided in the DGE GLM to adjust for the number of detected genes with
expression > 0.}

\item{return.groups}{A logical scalar that returns a \code{\link{data.frame}} of the
aggregated groups per single-cell. Cells that are members of non-DA neighbourhoods contain
\code{NA} values.}

\item{subset.nhoods}{A logical, integer or character vector indicating which neighbourhoods
to subset before aggregation and DGE testing.}

\item{na.function}{A valid NA action function to apply, should be one of
\code{na.fail, na.omit, na.exclude, na.pass}.}

\item{compute.new}{A logical scalar indicating whether to force computing a new neighbourhood
adjacency matrix if already present.}
}
\value{
A \code{data.frame} of DGE results containing a log fold change and adjusted
p-value for each aggregated group of neighbourhoods. If \code{return.groups} then
the return value is a list with the slots \code{groups} and \code{dge} containing the
aggregated neighbourhood groups per single-cell and marker gene results, respectively.

\emph{Warning}: If all neighbourhoods are grouped together, then it is impossible to
run \code{findNhoodMarkers}. In this (hopefully rare) instance, this function will return
a warning and return \code{NULL}.
}
\description{
This function will perform differential gene expression analysis on
differentially abundant neighbourhoods, by first aggregating adjacent and
concordantly DA neighbourhoods, then comparing cells \emph{between} these
aggregated groups. For differential gene experession based on an input design
\emph{within} DA neighbourhoods see \code{\link{testDiffExp}}.
}
\details{
Louvain clustering is applied to the neighbourhood graph. This graph is first modified
based on two criteria: 1) neighbourhoods share at least \code{overlap} number of cells,
and 2) the DA log fold change sign is concordant.
This behaviour can be modulated by setting \code{overlap} to be more or less stringent.
Additionally, a threshold on the log fold-changes can be set, such that \code{lfc.threshold}
is required to retain edges between adjacent neighbourhoods. Note: adjacent neighbourhoods will
never be merged with opposite signs.

Using a one vs. all approach, each aggregated group of cells is compared to all others
using the single-cell log normalized gene expression with a GLM
(for details see \code{\link[limma]{limma-package}}), or the single-cell counts using a
negative binomial GLM (for details see \code{\link[edgeR]{edgeR-package}}). When using
the latter it is recommended to set \code{gene.offset=TRUE} as this behaviour adjusts
the model offsets by the number of detected genes in each cell.
}
\examples{
library(SingleCellExperiment)
ux.1 <- matrix(rpois(12000, 5), ncol=400)
ux.2 <- matrix(rpois(12000, 4), ncol=400)
ux <- rbind(ux.1, ux.2)
vx <- log2(ux + 1)
pca <- prcomp(t(vx))

sce <- SingleCellExperiment(assays=list(counts=ux, logcounts=vx),
                            reducedDims=SimpleList(PCA=pca$x))
colnames(sce) <- paste0("Cell", seq_len(ncol(sce)))
milo <- Milo(sce)
milo <- buildGraph(milo, k=20, d=10, transposed=TRUE)
milo <- makeNhoods(milo, k=20, d=10, prop=0.3)
milo <- calcNhoodDistance(milo, d=10)

cond <- rep("A", ncol(milo))
cond.a <- sample(seq_len(ncol(milo)), size=floor(ncol(milo)*0.25))
cond.b <- setdiff(seq_len(ncol(milo)), cond.a)
cond[cond.b] <- "B"
meta.df <- data.frame(Condition=cond, Replicate=c(rep("R1", 132), rep("R2", 132), rep("R3", 136)))
meta.df$SampID <- paste(meta.df$Condition, meta.df$Replicate, sep="_")
milo <- countCells(milo, meta.data=meta.df, samples="SampID")

test.meta <- data.frame("Condition"=c(rep("A", 3), rep("B", 3)), "Replicate"=rep(c("R1", "R2", "R3"), 2))
test.meta$Sample <- paste(test.meta$Condition, test.meta$Replicate, sep="_")
rownames(test.meta) <- test.meta$Sample
da.res <- testNhoods(milo, design=~0 + Condition, design.df=test.meta[colnames(nhoodCounts(milo)), ])

nhood.dge <- findNhoodMarkers(milo, da.res, overlap=1, compute.new=TRUE)
nhood.dge

}
\author{
Mike Morgan & Emma Dann
}
