##' project_title UI Function
##'
##' @description A shiny Module allowing to specify the project name which will
##' be used for the outputs generated by WPM.
##'
##' @param id Internal parameters for {shiny}.
##' @noRd
mod_project_title_ui <- function(id){
    ns <- shiny::NS(id)
    shiny::fluidRow(
        shiny::column(width = 6,
                      shinydashboard::box(
                          status = "warning",
                          width = 12,
                          collapsible = TRUE,
                          solidHeader = FALSE,
                          title = shiny::h3("Please choose a Project name"),
                          shiny::textInput(inputId = ns("project_title"),
                                           label = NULL,
                                           value = "",
                                           placeholder = "my project title")
                      )
        ),
        shiny::column(width = 6,
                      shiny::fluidRow(
                          shiny::uiOutput(ns("title_check"))
                      )
        ) # end of column 2
    )
}

##' project_title Server Function
##'
##' @description Server part of the project_title module. Allows the
##' user to specify the plate dimensions and their number.
##'
##' @param input,output,session internal parameters of shiny
##'
##' @noRd
mod_project_title_server <- function(input, output, session){

    project_title <- shiny::reactive({
        input$project_title
    })

    output$title_check <- shiny::renderUI({
        if(project_title() == "" ){
            shinydashboard::infoBox(
                title = "Project title required",
                color = "red",
                width = 12,
                icon = shiny::icon("times")
            )
        }else if(stringr::str_length(project_title()) > 25){
            shinydashboard::infoBox(
                title = "Project title too long. Since it will be used to
                create the names of the output files, please make it explicit
                and as short as possible. The maximum length allowed is 20
                characters (not including spaces).",
                color = "red",
                width = 12,
                icon = shiny::icon("times")
            )
        }else if(stringr::str_detect(project_title(), stringr::coll("."))){
            shinydashboard::infoBox(
                title = "the '.' character is prohibited.",
                color = "red",
                width = 12,
                icon = shiny::icon("times")
            )
        }else{
            shinydashboard::infoBox(
                title = "",
                color = "green",
                width = 12,
                icon = shiny::icon("check-circle")
            )
        }
    })


    toReturn <- shiny::reactive({
        if(project_title() == ""
           & stringr::str_length(project_title()) > 25
           & stringr::str_detect(project_title(), stringr::coll("."))){
            return(NULL)
        }else{
            return(project_title())
        }
    })



    return(toReturn)
}
