% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods-MChromatograms.R
\name{plotChromatogramsOverlay}
\alias{plotChromatogramsOverlay}
\alias{plotChromatogramsOverlay,MChromatograms-method}
\alias{plotChromatogramsOverlay,XChromatograms-method}
\title{Plot multiple chromatograms into the same plot}
\usage{
\S4method{plotChromatogramsOverlay}{MChromatograms}(
  object,
  col = "#00000060",
  type = "l",
  main = NULL,
  xlab = "rtime",
  ylab = "intensity",
  xlim = numeric(),
  ylim = numeric(),
  stacked = 0,
  transform = identity,
  ...
)

\S4method{plotChromatogramsOverlay}{XChromatograms}(
  object,
  col = "#00000060",
  type = "l",
  main = NULL,
  xlab = "rtime",
  ylab = "intensity",
  xlim = numeric(),
  ylim = numeric(),
  peakType = c("polygon", "point", "rectangle", "none"),
  peakBg = NULL,
  peakCol = NULL,
  peakPch = 1,
  stacked = 0,
  transform = identity,
  ...
)
}
\arguments{
\item{object}{\code{\link[MSnbase:MChromatograms-class]{MSnbase::MChromatograms()}} or \code{\link[=XChromatograms]{XChromatograms()}} object.}

\item{col}{definition of the color in which the chromatograms should be
drawn. Can be of length 1 or equal to \code{nrow(object)} to plot each
overlayed chromatogram in a different color.}

\item{type}{\code{character(1)} defing the type of the plot. By default
(\code{type = "l"}) each chromatogram is drawn as a line.}

\item{main}{optional title of the plot. If not defined, the range of m/z
values is used.}

\item{xlab}{\code{character(1)} defining the x-axis label.}

\item{ylab}{\code{character(1)} defining the y-axis label.}

\item{xlim}{optional \code{numeric(2)} defining the x-axis limits.}

\item{ylim}{optional \code{numeric(2)} defining the y-axis limits.}

\item{stacked}{\code{numeric(1)} defining the part (proportion) of the y-axis to
use to \emph{stack} EICs depending on their m/z values. If \code{stacked = 0} (the
default) no stacking is performed. With \code{stacked = 1} half of the y-axis
is used for stacking and half for the intensity y-axis (i.e. the ratio
between stacking and intensity y-axis is 1:1). Note that if \code{stacking}
is different from 0 no y-axis and label are drawn.}

\item{transform}{\code{function} to transform the intensity values before
plotting. Defaults to \code{transform = identity} which plots the data as it
is. With \code{transform = log10} intensity values would be log10 transformed
before plotting.}

\item{...}{optional arguments to be passed to the plotting functions (see
help on the base R \code{plot} function.}

\item{peakType}{if \code{object} is a \code{XChromatograms} object: how chromatographic
peaks should be drawn: \code{peakType = "polygon"} (the default): label the
full chromatographic peak area, \code{peakType = "rectangle"}: indicate the
chromatographic peak by a rectangle and \code{peakType  = "point"}: label the
chromatographic peaks' apex position with a point.}

\item{peakBg}{if \code{object} is a \code{XChromatograms} object: definition of
background color(s) for each chromatographic peak. Has to be either of
length 1 or equal to the number of peaks in \code{object}. If not specified,
the peak will be drawn in the color defined by \code{col}.}

\item{peakCol}{if \code{object} is a \code{XChromatograms} object: definition of
color(s) for each chromatographic peak. Has to be either of length 1 or
equal to the number of peaks in \code{object}. If not specified, the peak will
be drawn in the color defined by \code{col}.}

\item{peakPch}{if \code{object} is a \code{XChromatograms} object: \emph{point character} to
be used to label the apex position of the chromatographic peak if
\code{peakType = "point"}.}
}
\value{
silently returns a \code{list} (length equal to \code{ncol(object)} of
\code{numeric} (length equal to \code{nrow(object)}) with the y position of
each EIC.
}
\description{
\code{plotOverlay} draws chromatographic peak data from multiple (different)
extracted ion chromatograms (EICs) into the same plot. This allows to
directly compare the peak shape of these EICs in the same sample. In
contrast to the \code{plot} function for \code{\link[MSnbase:MChromatograms-class]{MSnbase::MChromatograms()}} object,
which draws the data from the same EIC across multiple samples in the
same plot, this function draws the different EICs from the same sample
into the same plot.

If \code{plotChromatogramsOverlay} is called on a \code{XChromatograms} object any
present chromatographic peaks will also be highlighted/drawn depending on the
parameters \code{peakType}, \code{peakCol}, \code{peakBg} and \code{peakPch} (see also help on
the \code{plot} function for \code{XChromatogram()} object for details).
}
\examples{

## Load preprocessed data and extract EICs for some features.
library(xcms)
library(MSnbase)
xdata <- loadXcmsData()
data(xdata)
## Update the path to the files for the local system
dirname(xdata) <- c(rep(system.file("cdf", "KO", package = "faahKO"), 4),
                    rep(system.file("cdf", "WT", package = "faahKO"), 4))
## Subset to the first 3 files.
xdata <- filterFile(xdata, 1:3, keepFeatures = TRUE)

## Define features for which to extract EICs
fts <- c("FT097", "FT163", "FT165")
chrs <- featureChromatograms(xdata, features = fts)

plotChromatogramsOverlay(chrs)

## plot the overlay of EICs in the first sample
plotChromatogramsOverlay(chrs[, 1])

## Define a different color for each feature (row in chrs). By default, also
## all chromatographic peaks of a feature is labeled in the same color.
plotChromatogramsOverlay(chrs[, 1],
    col = c("#ff000040", "#00ff0040", "#0000ff40"))

## Alternatively, we can define a color for each individual chromatographic
## peak and provide this with the `peakBg` and `peakCol` parameters.
chromPeaks(chrs[, 1])

## Use a color for each of the two identified peaks in that sample
plotChromatogramsOverlay(chrs[, 1],
    col = c("#ff000040", "#00ff0040", "#0000ff40"),
    peakBg = c("#ffff0020", "#00ffff20"))

## Plotting the data in all samples.
plotChromatogramsOverlay(chrs,
    col = c("#ff000040", "#00ff0040", "#0000ff40"))

## Creating a "stacked" EIC plot: the EICs are placed along the y-axis
## relative to their m/z value. With `stacked = 1` the y-axis is split in
## half, the lower half being used for the stacking of the EICs, the upper
## half being used for the *original* intensity axis.
res <- plotChromatogramsOverlay(chrs[, 1], stacked = 1,
    col = c("#ff000040", "#00ff0040", "#0000ff40"))
## add horizontal lines for the m/z values of each EIC
abline(h = res[[1]], col = "grey", lty = 2)

## Note that this type of visualization is different than the conventional
## plot function for chromatographic data, which will draw the EICs for
## multiple samples into the same plot
plot(chrs)

## Converting the object to a MChromatograms without detected peaks
chrs <- as(chrs, "MChromatograms")

plotChromatogramsOverlay(chrs,
    col = c("#ff000040", "#00ff0040", "#0000ff40"))
}
\author{
Johannes Rainer
}
