% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster_fold_similarity.r
\name{clusterFoldSimilarity}
\alias{clusterFoldSimilarity}
\title{Calculate cluster similarity between clusters from different single cell samples.}
\usage{
clusterFoldSimilarity(
  scList = NULL,
  sampleNames = NULL,
  topN = 1,
  topNFeatures = 1,
  nSubsampling = 15,
  parallel = FALSE
)
}
\arguments{
\item{scList}{List. A list of Single Cell Experiments or Seurat objects. At least 2 are needed. The objects are expected to have cluster or label groups set as identity class.}

\item{sampleNames}{Character Vector. Specify the sample names, if not a number corresponding with its position on (scList).}

\item{topN}{Numeric. Specifies the number of target clusters with best similarity to report for each cluster comparison (default 1). If set to Inf, then all similarity values from all possible pairs of clusters are returned.}

\item{topNFeatures}{Numeric. Number of top features that explains the clusters similarity to report for each cluster comparison (default 1). If topN = Inf then topNFeatures is automatically set to 1.}

\item{nSubsampling}{Numeric. Number of random sampling of cells to achieve fold change stability (default 15).}

\item{parallel}{Boolean. Whether to use parallel computing using BiocParallel or not (default FALSE).}
}
\value{
The function returns a DataFrame containing the best top similarities between all possible pairs of single cell samples. Column values are:
\tabular{ll}{
   \code{similarityValue} \tab The top similarity value calculated between datasetL:clusterL and datasetR. \cr
   \tab \cr
   \code{w} \tab Weight associated with the similarity score value. \cr
   \tab \cr
   \code{datasetL} \tab Dataset left, the dataset/sample which has been used to be compared.  \cr
   \tab \cr
   \code{clusterL} \tab Cluster left, the cluster source from datasetL which has been compared. \cr
   \tab \cr
   \code{datasetR} \tab Dataset right, the dataset/sample used for comparison against datasetL. \cr
   \tab \cr
   \code{clusterR} \tab Cluster right, the cluster target from datasetR which is being compared with the clusterL from datasetL. \cr
   \tab \cr
   \code{topFeatureConserved} \tab The features (e.g.: genes, peaks...) that most contributed to the similarity between clusterL & clusterR. \cr
   \tab \cr
   \code{featureScore} \tab The similarity score contribution for the specific topFeatureConserved (e.g.: genes, peaks...). \cr
}
}
\description{
`clusterFoldSimilarity()` returns a dataframe containing the best top similarities between all possible pairs of single cell samples.
}
\details{
This function will calculate a similarity coeficient using the fold changes of shared features (e.g.:genes for a single-cell RNA-Seq, peaks for ATAC-Seq) among clusters, or user-defined-groups, from different samples/batches. The similarity coeficient
is calculated using the dotproduct of every pairwise combination of Fold Changes between a source cluster/group i from sample n and all the target clusters/groups in sample j.
}
\examples{
if (requireNamespace("Seurat") & requireNamespace("SeuratObject")){
library(ClusterFoldSimilarity)
library(Seurat)
library(SeuratObject)
# data dimensions
nfeatures <- 2000; ncells <- 400
# single-cell 1
counts <- matrix(rpois(n=nfeatures * ncells, lambda=10), nfeatures)
rownames(counts) <- paste0("gene",seq(nfeatures))
colnames(counts) <- paste0("cell",seq(ncells))
colData <- data.frame(cluster=sample(c("Cluster1","Cluster2","Cluster3"),size = ncells,replace = TRUE),
                     row.names=paste0("cell",seq(ncells)))
seu1 <- SeuratObject::CreateSeuratObject(counts = counts, meta.data = colData)
Idents(object = seu1) <- "cluster"
# single-cell 2
counts <- matrix(rpois(n=nfeatures * ncells, lambda=20), nfeatures)
rownames(counts) <- paste0("gene",seq(nfeatures))
colnames(counts) <- paste0("cell",seq(ncells))
colData <- data.frame(cluster=sample(c("Cluster1","Cluster2","Cluster3","Cluster4"),size = ncells,replace = TRUE),
                      row.names=paste0("cell",seq(ncells)))
seu2 <- SeuratObject::CreateSeuratObject(counts = counts, meta.data = colData)
Idents(object = seu2) <- "cluster"
# Create a list with the unprocessed single-cell datasets
singlecellObjectList <- list(seu1, seu2)

similarityTable <- clusterFoldSimilarity(scList=singlecellObjectList, sampleNames = c("sc1","sc2"))
head(similarityTable)
}

}
\author{
Oscar Gonzalez-Velasco
}
