% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/resultAnnotations.R
\name{potentialImpactAnnotations}
\alias{potentialImpactAnnotations}
\alias{annotateIntronReferenceOverlap}
\alias{annotatePotentialImpact}
\alias{flagBlacklistRegions}
\title{Additional result annotations}
\usage{
annotateIntronReferenceOverlap(fds, txdb, BPPARAM = bpparam())

annotatePotentialImpact(
  result,
  txdb,
  fds,
  addPotentialImpact = TRUE,
  addUTRoverlap = TRUE,
  minoverlap = 5,
  BPPARAM = bpparam()
)

flagBlacklistRegions(
  result,
  blacklist_regions = NULL,
  assemblyVersion = c("hg19", "hg38"),
  minoverlap = 5
)
}
\arguments{
\item{fds}{A FraserDataSet}

\item{txdb}{A txdb object providing the reference annotation.}

\item{BPPARAM}{For controlling parallelization behavior. Defaults to 
\code{bpparam()}.}

\item{result}{A result table as generated by FRASER, including the column 
\code{annotatedJunction} as generated by the function
\code{annotateIntronReferenceOverlap}.}

\item{addPotentialImpact}{Logical, indicating if the type of the potential 
impact should be added to the results table. Defaults to \code{TRUE}.}

\item{addUTRoverlap}{Logical, indicating if the overlap with UTR regions
should checked and added to the results table. Defaults to \code{TRUE}.}

\item{minoverlap}{Integer value defining the number of base pairs around the
splice site that need to overlap with UTR or blacklist region, 
respectivly, to be considered matching. Defaults to 5 bp.}

\item{blacklist_regions}{A BED file that contains the blacklist regions. 
If \code{NULL} (default), the BED files that are packaged with FRASER 
are used (see Details for more information).}

\item{assemblyVersion}{Indicates the genome assembly version of the intron 
coordinates. Only used if blacklist_regions is NULL. For other versions, 
please provide the BED file containing the blacklist regions directly.}
}
\value{
An annotated FraserDataSet or results table, respectively
}
\description{
These functions work on the result table and add additional 
    annotations to the reported introns: the type of potential impact on 
    splicing (e.g. exon skipping, exon truncation, ...), potential occurence 
    of frameshift, overlap with UTR regions as well as a flag for introns 
    that are located in blacklist regions of the genome.
    
\code{\link{annotateIntronReferenceOverlap}} adds basic annotations to the 
    fds for each intron based on the overlap of the intron's location with 
    the reference annotation. Has to be run before the result table is 
    created so that the new column can be included in it (see examples).

\code{\link{annotatePotentialImpact}} annotates each intron in the results 
    table with the type of potential impact on splicing and potential 
    occurence of frameshift (likely, unlikely, inconclusive). Can also 
    calculate overlap with annotated UTR regions. Potential impact can be: 
    annotatedIntron_increasedUsage, annotatedIntron_reducedUsage, 
    exonTruncation, exonElongation, exonTruncation&Elongation, 
    exonSkipping, splicingBeyondGene, 
    multigenicSplicing, downstreamOfNearestGene, upstreamOfNearestGene, 
    complex (everything else).     
    Splice sites (theta metric) annotations indicate how the splice site is 
    located with respect to the reference annotation. The annotated types 
    are: annotatedSpliceSite, exonicRegion, intronicRegion.

\code{\link{flagBlacklistRegions}} flags introns in the results table on 
    whether or not they are located in a blacklist region of the genome. By
    default, the blacklist regions as reported in 
    \cite{Amemiya, Kundaje & Boyle (2019)} and downloaded from 
    \href{https://www.encodeproject.org/annotations/ENCSR636HFF/}{here} 
    are used.
}
\section{Functions}{
\itemize{
\item \code{annotateIntronReferenceOverlap()}: This method calculates basic annotations 
based on overlap with the reference annotation (start, end, none, both) 
for the full fds. The overlap type is added as a new column 
\code{annotatedJunction} in \code{mcols(fds)}.

\item \code{annotatePotentialImpact()}: This method annotates the splice event 
type to junctions in the given results table.

\item \code{flagBlacklistRegions()}: This method flags all introns and 
splice sites in the given results table for which at least one splice 
site (donor or acceptor) is located in a blacklist region. Blacklist 
regions of the genome are determined from the provided BED file.

}}
\examples{
  # get data, fit and compute p-values and z-scores
  fds <- createTestFraserDataSet()
  
  # load reference annotation
  library(TxDb.Hsapiens.UCSC.hg19.knownGene)
  txdb <- TxDb.Hsapiens.UCSC.hg19.knownGene 
    
  # add basic annotations for overlap with the reference annotation
  # run this function before creating the results table
  fds <- annotateIntronReferenceOverlap(fds, txdb) 

  # extract results: for this small example dataset, no cutoffs used 
  # to get some results 
  res <- results(fds, padjCutoff=NA, deltaPsiCutoff=NA)
  
  # annotate the type of potential impact on splicing and UTR overlap
  res <- annotatePotentialImpact(result=res, txdb=txdb, fds=fds)
  
  # annotate overlap with blacklist regions
  res <- flagBlacklistRegions(result=res, assemblyVersion="hg19")
  
  # show results table containing additional annotations
  res
  
}
