% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCMC_chain.R
\name{run_metropolis_MCMC_betas}
\alias{run_metropolis_MCMC_betas}
\title{Run Metropolis-Hastings MCMC for HMRFHiC Model}
\usage{
run_metropolis_MCMC_betas(
  N,
  gamma_prior,
  iterations,
  x_vars,
  y,
  use_data_priors,
  user_fixed_priors = NULL,
  dist = "ZIP",
  epsilon = NULL,
  distance_metric = "manhattan",
  size_start = NULL,
  theta_start = NULL
)
}
\arguments{
\item{N}{Integer specifying the dimension of the lattice (\eqn{N \times N}).}

\item{gamma_prior}{Numeric. Initial (prior) value for the interaction parameter \eqn{\gamma} in the Potts model. Set at 0.3 as default.}

\item{iterations}{Integer. Number of MCMC iterations to run.}

\item{x_vars}{A list of covariates used as predictors. Must contain named elements "distance", "GC", "TES", and "ACC", each a list of \eqn{N \times N} matrices.}

\item{y}{An \eqn{N \times N} numeric matrix of observed interaction counts, with elements corresponding to locus pairs (i,j).}

\item{use_data_priors}{Logical. If \code{TRUE}, uses data-driven priors for each component. If \code{FALSE}, \code{user_fixed_priors} must be provided.}

\item{user_fixed_priors}{(Optional) A list of user-specified priors for model components if \code{use_data_priors = FALSE}. Each component’s priors should include means and standard deviations.}

\item{dist}{Character string specifying the distribution family: \code{"Poisson"}, \code{"NB"} (Negative Binomial), \code{"ZIP"} (Zero-Inflated Poisson, default), or \code{"ZINB"} (Zero-Inflated Negative Binomial).}

\item{epsilon}{(Optional) Numeric tolerance for the ABC update of \eqn{\gamma}. If \code{NULL}, computed dynamically from data.}

\item{distance_metric}{Character string specifying the distance metric for ABC: \code{"manhattan"} (default) or \code{"euclidean"}.}

\item{size_start}{(Optional) Numeric vector of length 3 specifying initial size (overdispersion) parameters for each component if \code{dist} is "NB" or "ZINB".}

\item{theta_start}{(Optional) Numeric initial value for \eqn{\theta}, the zero-inflation parameter in ZIP/ZINB models. Defaults to 0.5 if \code{NULL} and \code{dist} is ZIP/ZINB.}
}
\value{
A list containing:
\itemize{
  \item \code{chains}: List of three \eqn{(iterations+1) \times 5} matrices, each storing parameter chains for one of the three components.
  \item \code{gamma}: Numeric vector of length \eqn{iterations+1}, the chain of \eqn{\gamma} values.
  \item \code{theta}: Numeric vector of length \eqn{iterations+1}, the chain of \eqn{\theta} values (unchanged if not ZIP/ZINB).
  \item \code{size}: \eqn{3 \times (iterations+1)} matrix of size (overdispersion) parameters if \code{dist} is NB or ZINB; otherwise unused.
}
}
\description{
Runs a Markov Chain Monte Carlo (MCMC) algorithm using Metropolis-Hastings to sample parameters for the Hidden Markov Random Field (HMRF) model in HMRFHiC.
Integrates proposal distributions, acceptance/rejection steps based on posterior calculations, and Approximate Bayesian Computation (ABC) updates for the \eqn{\gamma} parameter.
Supports data-driven or user-defined priors for covariates.
}
\details{
The algorithm proceeds as follows:
\enumerate{
  \item \strong{Initialization:}
    Initializes the component assignment matrix \eqn{z} based on \code{y} and sets starting values for parameters (\eqn{\gamma}, \eqn{\theta}, and \eqn{\text{size}} if applicable).
  \item \strong{MCMC Updates per Iteration:}
    \itemize{
      \item Updates latent assignments \eqn{z} by proposing new states and computing posterior probabilities using internal functions.
      \item Updates model parameters (betas) for each component via Metropolis-Hastings:
        \itemize{
          \item Proposes new values from normal distributions.
          \item Computes posterior probabilities and accepts/rejects proposals.
        }
      \item Updates overdispersion \code{size} or zero-inflation \eqn{\theta} parameters if applicable.
      \item Updates \eqn{\gamma} via ABC:
        \itemize{
          \item Proposes a new \eqn{\gamma} and simulates synthetic data.
          \item Compares to observed data using the specified \code{distance_metric} and \eqn{\epsilon}.
          \item Accepts/rejects based on whether the distance is below \eqn{\epsilon}.
        }
    }
  \item \strong{Adaptive Tuning:}
    Tunes proposal standard deviations to achieve a target acceptance rate.
}
}
\examples{

N <- 10
gamma_prior <- 0.5
iterations <- 10
x_vars <- list(
  distance = list(matrix(runif(N * N, 0, 10), nrow = N)),
  GC = list(matrix(runif(N * N, 0, 1), nrow = N)),
  TES = list(matrix(runif(N * N, 0, 2), nrow = N)),
  ACC = list(matrix(runif(N * N, 0, 5), nrow = N))
)
y <- matrix(rpois(N * N, lambda = 5), nrow = N)
results <- run_metropolis_MCMC_betas(
  N = N,
  gamma_prior = gamma_prior,
  iterations = iterations,
  x_vars = x_vars,
  y = y,
  use_data_priors = TRUE,
  dist = "ZIP"
)
plot(results$gamma, type = "l")


}
