% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/processStudy.R
\encoding{UTF-8}
\name{pruningSample}
\alias{pruningSample}
\title{Compute the list of pruned SNVs for a specific profile using the
information from the Reference GDS file and a linkage disequilibrium
analysis}
\usage{
pruningSample(
  gdsReference,
  method = c("corr", "r", "dprime", "composite"),
  currentProfile,
  studyID,
  listSNP = NULL,
  slideWindowMaxBP = 500000L,
  thresholdLD = sqrt(0.1),
  np = 1L,
  verbose = FALSE,
  chr = NULL,
  superPopMinAF = NULL,
  keepPrunedGDS = TRUE,
  pathProfileGDS = NULL,
  keepFile = FALSE,
  pathPrunedGDS = ".",
  outPrefix = "pruned"
)
}
\arguments{
\item{gdsReference}{an object of class \link[gdsfmt]{gds.class} (a GDS
file), the 1 KG GDS file (reference data set).}

\item{method}{a \code{character} string that represents the method that will
be used to calculate the linkage disequilibrium in the
\code{\link[SNPRelate]{snpgdsLDpruning}}() function. The 4 possible values
are: "corr", "r", "dprime" and "composite". Default: \code{"corr"}.}

\item{currentProfile}{a \code{character} string
corresponding to the profile identifier used in LD pruning done by the
\code{\link[SNPRelate]{snpgdsLDpruning}}() function. A Profile GDS file
corresponding to the profile identifier must exist and be located in the
\code{pathProfileGDS} directory.}

\item{studyID}{a \code{character} string corresponding to the study
identifier used in the \code{\link[SNPRelate]{snpgdsLDpruning}} function.
The study identifier must be present in the Profile GDS file.}

\item{listSNP}{a \code{vector} of SNVs identifiers specifying selected to
be passed the the pruning function;
if \code{NULL}, all SNVs are used in the
\code{\link[SNPRelate]{snpgdsLDpruning}} function. Default: \code{NULL}.}

\item{slideWindowMaxBP}{a single positive \code{integer} that represents
the maximum basepairs (bp) in the sliding window. This parameter is used
for the LD pruning done in the \code{\link[SNPRelate]{snpgdsLDpruning}}
function.
Default: \code{500000L}.}

\item{thresholdLD}{a single \code{numeric} value that represents the LD
threshold used in the \code{\link[SNPRelate]{snpgdsLDpruning}} function.
Default: \code{sqrt(0.1)}.}

\item{np}{a single positive \code{integer} specifying the number of
threads to be used. Default: \code{1L}.}

\item{verbose}{a \code{logicial} indicating if information is shown
during the process in the \code{\link[SNPRelate]{snpgdsLDpruning}}
function.  Default: \code{FALSE}.}

\item{chr}{a \code{character} string representing the chromosome where the
selected SNVs should belong. Only one chromosome can be handled. If
\code{NULL}, the chromosome is not used as a filtering criterion.
Default: \code{NULL}.}

\item{superPopMinAF}{a single positive \code{numeric} representing the
minimum allelic frequency used to select the SNVs. If \code{NULL}, the
allelic frequency is not used as a filtering criterion. Default: \code{NULL}.}

\item{keepPrunedGDS}{a \code{logicial} indicating if the information about
the pruned SNVs should be added to the GDS Sample file.
Default: \code{TRUE}.}

\item{pathProfileGDS}{a \code{character} string representing the directory
where the Profile GDS files will be created. The directory must exist.}

\item{keepFile}{a \code{logical} indicating if RDS files containing the
information about the pruned SNVs must be
created. Default: \code{FALSE}.}

\item{pathPrunedGDS}{a \code{character} string representing an existing
directory. The directory must exist. Default: \code{"."}.}

\item{outPrefix}{a \code{character} string that represents the prefix of the
RDS files that will be generated. The RDS files are only generated when
the parameter \code{keepFile}=\code{TRUE}. Default: \code{"pruned"}.}
}
\value{
The function returns \code{0L} when successful.
}
\description{
This function computes the list of pruned SNVs for a
specific profile. When
a group of SNVs are in linkage disequilibrium, only one SNV from that group
is retained. The linkage disequilibrium is calculated with the
\code{\link[SNPRelate]{snpgdsLDpruning}}() function. The initial list of
SNVs that are passed to the \code{\link[SNPRelate]{snpgdsLDpruning}}()
function can be specified by the user.
}
\examples{

## Required library for GDS
library(gdsfmt)

## Path to the demo Reference GDS file is located in this package
dataDir <- system.file("extdata/tests", package="RAIDS")
fileGDS <- file.path(dataDir, "ex1_good_small_1KG.gds")

## The data.frame containing the information about the study
## The 3 mandatory columns: "study.id", "study.desc", "study.platform"
## The entries should be strings, not factors (stringsAsFactors=FALSE)
studyDF <- data.frame(study.id = "MYDATA",
                        study.desc = "Description",
                        study.platform = "PLATFORM",
                        stringsAsFactors = FALSE)

## The data.frame containing the information about the samples
## The entries should be strings, not factors (stringsAsFactors=FALSE)
samplePED <- data.frame(Name.ID = c("ex1", "ex2"),
                    Case.ID = c("Patient_h11", "Patient_h12"),
                    Diagnosis = rep("Cancer", 2),
                    Sample.Type = rep("Primary Tumor", 2),
                    Source = rep("Databank B", 2), stringsAsFactors = FALSE)
rownames(samplePED) <- samplePED$Name.ID

## Temporary Profile GDS file
profileFile <- file.path(tempdir(), "ex1.gds")

## Copy the Profile GDS file demo that has not been pruned yet
file.copy(file.path(dataDir, "ex1_demo.gds"), profileFile)

## Open 1KG file
gds1KG <- snpgdsOpen(fileGDS)

## Compute the list of pruned SNVs for a specific profile 'ex1'
## and save it in the Profile GDS file 'ex1.gds'
pruningSample(gdsReference=gds1KG, currentProfile=c("ex1"),
              studyID = studyDF$study.id, pathProfileGDS=tempdir())

## Close the Reference GDS file (important)
closefn.gds(gds1KG)

## Check content of Profile GDS file
## The 'pruned.study' entry should be present
content <- openfn.gds(profileFile)
content

## Close the Profile GDS file (important)
closefn.gds(content)

## Remove Profile GDS file (created for demo purpose)
unlink(profileFile, force=TRUE)


}
\author{
Pascal Belleau, Astrid Deschênes and Alexander Krasnitz
}
