% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tune.block.plsda.R
\name{tune.block.plsda}
\alias{tune.block.plsda}
\title{Tuning function for block.plsda method (N-integration with Discriminant Analysis)}
\usage{
tune.block.plsda(
  X,
  Y,
  indY,
  ncomp = 2,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  design,
  scale = TRUE,
  validation = "Mfold",
  folds = 10,
  nrepeat = 1,
  signif.threshold = 0.01,
  dist = "all",
  weighted = TRUE,
  progressBar = FALSE,
  light.output = TRUE,
  BPPARAM = SerialParam(),
  seed = NULL,
  ...
)
}
\arguments{
\item{X}{A named list of data sets (called 'blocks') measured on the same 
samples. Data in the list should be arranged in matrices, samples x variables, 
with samples order matching in all data sets.}

\item{Y}{a factor or a class vector for the discrete outcome.}

\item{indY}{To supply if \code{Y} is missing, indicates the position of 
the matrix response in the list \code{X}.}

\item{ncomp}{the number of components to include in the model. Default to 2.
Applies to all blocks.}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{design}{numeric matrix of size (number of blocks in X) x (number of
blocks in X) with values between 0 and 1. Each value indicates the strenght
of the relationship to be modelled between two blocks; a value of 0
indicates no relationship, 1 is the maximum value. Alternatively, one of
c('null', 'full') indicating a disconnected or fully connected design,
respecively, or a numeric between 0 and 1 which will designate all
off-diagonal elements of a fully connected design (see examples in
\code{block.splsda}). If \code{Y} is provided instead of \code{indY}, the
\code{design} matrix is changed to include relationships to \code{Y}.}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{validation}{character.  What kind of (internal) validation to use,
matching one of \code{"Mfold"} or \code{"loo"} (see below). Default is
\code{"Mfold"}.}

\item{folds}{the folds in the Mfold cross-validation. See Details.}

\item{nrepeat}{Number of times the Cross-Validation process is repeated.}

\item{signif.threshold}{numeric between 0 and 1 indicating the significance
threshold required for improvement in error rate of the components. Default
to 0.01.}

\item{dist}{Distance metric. Should be a subset of "max.dist", "centroids.dist", "mahalanobis.dist" or "all".
Default is "all"}

\item{weighted}{tune using either the performance of the Majority vote or
the Weighted vote.}

\item{progressBar}{by default set to \code{TRUE} to output the progress bar
of the computation.}

\item{light.output}{if set to FALSE, the prediction/classification of each
sample for each of \code{test.keepX} and each comp is returned.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object indicating the type
of parallelisation. See examples.}

\item{seed}{set a number here if you want the function to give reproducible outputs. 
Not recommended during exploratory analysis. Note if RNGseed is set in 'BPPARAM', this will be overwritten by 'seed'.}

\item{...}{Optional arguments:
\itemize{
 \item \bold{seed} Integer. Seed number for reproducible parallel code.
 Default is \code{NULL}.
}
run in parallel when repeating the cross-validation, which is usually the
most computationally intensive process. If there is excess CPU, the
cross-vaidation is also parallelised on *nix-based OS which support
\code{mclapply}.
Note that the argument 'scheme' has now been hardcoded to 'horst' and 'init' to 'svd.single'.}
}
\value{
returns:
\item{error.rate}{Prediction error rate for each block of \code{object$X}
and each \code{dist}} \item{error.rate.per.class}{Prediction error rate for
each block of \code{object$X}, each \code{dist} and each class}
\item{predict}{Predicted values of each sample for each class, each block
and each component} \item{class}{Predicted class of each sample for each
block, each \code{dist}, each component and each nrepeat} \item{features}{a
list of features selected across the folds (\code{$stable.X} and
\code{$stable.Y}) for the \code{keepX} and \code{keepY} parameters from the
input object.} \item{AveragedPredict.class}{if more than one block, returns
the average predicted class over the blocks (averaged of the \code{Predict}
output and prediction using the \code{max.dist} distance)}
\item{AveragedPredict.error.rate}{if more than one block, returns the
average predicted error rate over the blocks (using the
\code{AveragedPredict.class} output)} \item{WeightedPredict.class}{if more
than one block, returns the weighted predicted class over the blocks
(weighted average of the \code{Predict} output and prediction using the
\code{max.dist} distance). See details for more info on weights.}
\item{WeightedPredict.error.rate}{if more than one block, returns the
weighted average predicted error rate over the blocks (using the
\code{WeightedPredict.class} output.)} \item{MajorityVote}{if more than one
block, returns the majority class over the blocks. NA for a sample means that
there is no consensus on the predicted class for this particular sample over
the blocks.} \item{MajorityVote.error.rate}{if more than one block, returns
the error rate of the \code{MajorityVote} output}
\item{WeightedVote}{if more than one block, returns the weighted majority
class over the blocks. NA for a sample means that there is no consensus on
the predicted class for this particular sample over the blocks.}
\item{WeightedVote.error.rate}{if more than one block, returns the error
rate of the \code{WeightedVote} output} \item{weights}{Returns the weights
of each block used for the weighted predictions, for each nrepeat and each
fold} \item{choice.ncomp}{For supervised models; returns the optimal number
of components for the model for each prediction distance using one-sided
t-tests that test for a significant difference in the mean error rate (gain
in prediction) when components are added to the model. See more details in
Rohart et al 2017 Suppl. For more than one block, an optimal ncomp is
returned for each prediction framework.}
}
\description{
Computes M-fold or Leave-One-Out Cross-Validation scores based on a
user-input grid to determine the optimal parameters for
method \code{block.plsda}.
}
\details{
This tuning function should be used to tune the number of components in the \code{block.plsda} function (N-integration with Discriminant Analysis).

M-fold or LOO cross-validation is performed with stratified subsampling
where all classes are represented in each fold.

If \code{validation = "Mfold"}, M-fold cross-validation is performed. The
number of folds to generate is to be specified in the argument \code{folds}.

If \code{validation = "loo"}, leave-one-out cross-validation is performed.
By default \code{folds} is set to the number of unique individuals.

More details about the prediction distances in \code{?predict} and the
supplemental material of the mixOmics article (Rohart et al. 2017). Details
about the PLS modes are in \code{?pls}.

BER is appropriate in case of an unbalanced number of samples per class as
it calculates the average proportion of wrongly classified samples in each
class, weighted by the number of samples in each class. BER is less biased
towards majority classes during the performance assessment.
}
\examples{
data("breast.TCGA")

# X data - list of mRNA and miRNA
X <- list(mrna = breast.TCGA$data.train$mrna, mirna = breast.TCGA$data.train$mirna,
          protein = breast.TCGA$data.train$protein)

# Y data - single data set of proteins
Y <- breast.TCGA$data.train$subtype

# subset the X and Y data to speed up computation in this example
set.seed(100)
subset <- mixOmics:::stratified.subsampling(breast.TCGA$data.train$subtype, folds = 3)[[1]][[1]]
X <- lapply(X, function(omic) omic[subset,])
Y <- Y[subset]

# set up a full design where every block is connected
# could also consider other weights, see our mixOmics manuscript
design = matrix(1, ncol = length(X), nrow = length(X),
                dimnames = list(names(X), names(X)))
diag(design) =  0
design

## Tune number of components to keep - use all distance metrics
tune_res <- tune.block.plsda(X, Y, design = design,
                             ncomp = 5,
                             nrepeat = 3,
                             seed = 13,
                             dist = c("all"))

plot(tune_res)
tune_res$choice.ncomp # 3 components best for max.dist, 1 for centroids.dist


## Tune number of components to keep - use weighted vote rather than majority vote
tune_res <- tune.block.plsda(X, Y, design = design,
                             ncomp = 5,
                             nrepeat = 3,
                             seed = 13,
                             dist = c("all"),
                             weighted = FALSE)
tune_res$weights

## Tune number of components to keep - plot just max.dist
tune_res <- tune.block.plsda(X, Y, design = design,
                             ncomp = 5,
                             nrepeat = 3,
                             seed = 13,
                             dist = c("max.dist"))
plot(tune_res)
}
\references{
Method:

Singh A., Gautier B., Shannon C., Vacher M., Rohart F., Tebbutt S. and Lê
Cao K.A. (2016). DIABLO: multi omics integration for biomarker discovery.

mixOmics article:

Rohart F, Gautier B, Singh A, Lê Cao K-A. mixOmics: an R package for 'omics
feature selection and multiple data integration. PLoS Comput Biol 13(11):
e1005752
}
\seealso{
\code{\link{block.splsda}} and http://www.mixOmics.org for more
details.
}
\author{
Florian Rohart, Amrit Singh, Kim-Anh Lê Cao, AL J Abadi
}
\keyword{multivariate}
\keyword{regression}
