#' Plots the distribution of genomic features in the genome and those that 
#' overlap with sRNA clusters 
#'
#' @description Using the RNAfeatures() function, this function plots the 
#' percentage distribution of genomic features in the provided genome annotation 
#' and percentage the distibution of genomic features which overlap sRNA 
#' clusters. This is illustrated as a stacked bar plot. 
#'
#'
#' @details
#' `RNAfeatures` calculates the number or percentage of sRNA clusters which 
#' overlap with genomic features based on their genomic coordinates. 
#'
#' 
#' @seealso [mobileRNA::RNAmergeAnnotations()] to merge 2 GFF files into 1. 
#'
#' @param data data.frame; generated by [mobileRNA::RNAimport()]
#'
#' @param annotation path; URL or connection to a GFFFile object. A genome
#' reference annotation file (.gff/.gff1/.gff2/.gff3). Can be in compressed 
#' format (gzip).
#'
#' @param repeats path; URL or connection to a GFFFile object. A genome
#' reference annotation file, which only contains information on repeat
#' sequences in the genome (.gff/.gff1/.gff2/.gff3). By default, this is not
#' required, however if there is a specific repeats annotation file for the
#' genome it is suggested to supply it. Can be in compressed format (gzip).
#' 
#' @param promoterRegions numeric; defines the upstream promoter region of 
#' genes. Default is 1000, which refers to promoters set at 1Kb upstream of
#' genes
#' 
#'
#'@param repeat.type character; features type in `annotation` file to represent
#'repeats or transposable elements when `repeats` not supplied. Default is 
#'`c("transposable_element", transposable_element_gene")` which represent the 
#'transposable element features in the TAIR10 genome annotation. 
#'
#'
#'@param brewerPalette vector; colour scales from ColorBrewer to support the 
#'ggplot2::scale_fill_brewer function. Default is "Spectral". 
#'
#'@param x.axis.text vector; labs to represent the genome and the dataset 
#'bars in the plot. Default is x.axis.text=c("Genome", "Dataset"). 
#'
#'
#'@param legend.position character; position of legend. Either "none", "left", 
#'"right", "bottom", "top", "inside". 
#'
#' @return Returns a table containing the number or percentage of overlaps in
#' the supplied sRNA data set with specific regions in the genome annotation 
#' such as genes, repeats, introns, exons.
#'
#' @examples
#' data("sRNA_data")
#' features_plot <- plotRNAfeatures(data = sRNA_data,
#'                        annotation = system.file("extdata",
#'                        "reduced_chr2_Tomato.gff.gz", package="mobileRNA"))
#'                        
#'@importFrom rtracklayer import
#'@importFrom GenomicRanges setdiff
#'@importFrom GenomicRanges findOverlaps
#'@importFrom S4Vectors queryHits
#'@importFrom stats start
#'@importFrom stats end
#'@importFrom BiocGenerics strand
#'@importFrom dplyr select
#'@importFrom dplyr mutate
#'@importFrom dplyr filter
#'@importFrom GenomicRanges makeGRangesFromDataFrame
#'@importFrom scales label_percent
#'@importFrom dplyr %>%
#'@importFrom IRanges overlapsAny
#'@importFrom ggplot2 ggplot
#'@importFrom ggplot2 geom_bar
#'@importFrom ggplot2 labs
#'@importFrom ggplot2 scale_fill_brewer
#'@importFrom ggplot2 guides
#'@importFrom ggplot2 theme
#'@importFrom ggplot2 element_blank
#'@importFrom ggplot2 element_text
#'@importFrom ggplot2 element_line
#'@importFrom ggplot2 margin
#'@importFrom grid unit
#'@export

plotRNAfeatures <- function(data, annotation,
                            repeats = NULL,
                            promoterRegions = 1000, 
                            repeat.type = NULL, 
                            brewerPalette = "Spectral", 
                            x.axis.text =  c("Genome", "Dataset"), 
                            legend.position = "bottom" ){
  
  data_res <- RNAfeatures(data = data,
                                     annotation = annotation,
                                     repeats = repeats, 
                                     percentage = F)
  
  vec1 <- rep(rownames(as.data.frame(data_res)), 2)
  vec2 <- rep("Genome", 7)
  vec3 <- rep("Dataset", 7)
  vec4 <- as.data.frame(data_res)$Genome
  vec5 <- as.data.frame(data_res)$Dataset
  
  sum_vec4 <- sum(as.data.frame(data_res)$Genome)
  sum_vec5 <- sum(as.data.frame(data_res)$Dataset)
  
  perc_vec4 <- round((vec4/sum_vec4) *100, digits = 3)
  perc_vec5 <- round((vec5/sum_vec5) *100, digits = 3)
  
  df <- data.frame(Origin = c(vec2, vec3), 
                   variable = vec1, 
                   value = c(vec4, vec5),
                   percentage = c(perc_vec4, perc_vec5) )
  df[is.na(df)] <- 0
  
 df$Origin <- factor(df$Origin, 
                     levels = c("Genome", "Dataset"), 
                     labels =  x.axis.text)
  
  p <- ggplot2::ggplot(df, 
                       ggplot2::aes(x = Origin, 
                                    y = percentage, fill = variable)) +
    ggplot2::geom_bar(stat = "identity", alpha = 0.9, color = "black") +
    ggplot2::labs(x = "", y = "Percentage (%)", fill = "Genomic Feature") +
    ggplot2::theme_bw()+
    ggplot2::scale_fill_brewer(palette = brewerPalette ,direction = -1)+  
    ggplot2::guides(fill = ggplot2::guide_legend(override.aes = 
                                                   list(shape = 20, size = 4)))+
    ggplot2::theme(
      strip.background = ggplot2::element_blank(),
      strip.placement = "outside",
      panel.grid = ggplot2::element_blank(),
      axis.text.x = ggplot2::element_text(color="black", 
                                          size = 15,face = "bold", 
                                          margin = ggplot2::margin(t = 10, 
                                                                   b = 4)),
      axis.text.y = ggplot2::element_text(color="black", size = 15,
                                          face = "bold", 
                                          margin = ggplot2::margin(r = 10)) ,
      panel.grid.major.x = ggplot2::element_line( linewidth=.1, color="grey",
                                             linetype = 2 ),
      panel.grid.major.y = ggplot2::element_line( linewidth=.1, color="grey", 
                                             linetype = 2 ),
      legend.position = legend.position,
      legend.text = ggplot2::element_text(size=14, 
                                          margin = ggplot2::margin(7,7,7,7)),
      legend.title = ggplot2::element_text(size = 14.5, face = "bold"),
      axis.title.y = ggplot2::element_text(margin = ggplot2::margin(r = 20), size = 20, 
                                      face = "bold"),
      axis.title.x = ggplot2::element_text(size = 20, face = "bold"),
      strip.text.x = ggplot2::element_text(size = 16,face="italic" , 
                                      margin = ggplot2::margin(b = 10)),
      strip.text.y = ggplot2::element_text(size = 16,face="italic" , 
                                      margin = ggplot2::margin(l = 10)),
      plot.margin = grid::unit(c(1,1,1,1), units = "cm"))
  
  return(p)
  
}
