#' @title Save to disk a plot of all ROI EIC and detected feature range
#'
#' @description Plot and save a \code{.png} of all ROI (x is RT, y is
#' intensity), with the matching detected peak rt and peakwidth under it.
#'
#' @param ROIsDataPoint (list) a list of \code{data.frame} of raw data points
#' for each ROI (retention time 'rt', mass 'mz' and intensity 'int' (as column)
#' of each raw data points (as row)).
#' @param curveFit (list) a list of \code{peakPantheR_curveFit} or NA for each
#' ROI
#' @param foundPeakTable (data.frame) \code{data.frame} as generated by
#' \link{findTargetFeatures}, with features as rows and peak properties as
#' columns. The following columns are mandatory: \code{cpdID}, \code{cpdName},
#' \code{rt}, \code{rtmin}, \code{rtmax}, \code{mzmin}, \code{mzmax}.
#' @param savePath (str) Full path to save a \emph{.png} of all ROI EICs,
#' expect \code{'filepath/filename.png'}.
#' @param width (float) Width in cm for a single ROI plot (if more than one plot
#' in total, 2 columns will be used). dpi set to a 100.
#' @param height (float) height in a cm for a single ROI plot. dpi set to 100
#' @param verbose (bool) if TRUE message progress
#'
#' @return None
saveSingleFileMultiEIC <- function(ROIsDataPoint, curveFit, foundPeakTable,
    savePath, width = 15, height = 15, verbose = TRUE) {

    # check input
    nbROI <- length(ROIsDataPoint)
    if (nbROI != dim(foundPeakTable)[1]) {
        stop("Number of ROI datapoints in \"ROIsDataPoint\" (", nbROI,
            ") and features in \"foundPeakTable\" (",
            dim(foundPeakTable)[1], ") do not match!")
    }
    if (nbROI != length(curveFit)) {
        stop("Number of ROI datapoints in \"ROIsDataPoint\" (", nbROI,
            ") and fitted curves in \"curveFit\" (",
            length(curveFit), ") do not match!")
    }
    # Generate each ROI plot
    p_all <- vector("list", nbROI)
    for (i in seq_len(nbROI)) {
        p_all[[i]] <- plotEICDetectedPeakwidth(
            ROIDataPointSampleList = list(ROIsDataPoint[[i]]),
            cpdID = foundPeakTable$cpdID[i],
            cpdName = foundPeakTable$cpdName[i],
            rt = foundPeakTable$rt[i],
            rtMin = foundPeakTable$rtMin[i], rtMax = foundPeakTable$rtMax[i],
            mzMin = foundPeakTable$mzMin[i], mzMax = foundPeakTable$mzMax[i],
            ratio = 0.85,
            curveFitSampleList = list(curveFit[[i]]),
            sampling = 250,
            sampleColour = "black",
            verbose = verbose)
    }
    # Set save parameters
    targetFolder <- dirname(savePath)
    targetFile <- basename(savePath)
    nSubplot <- length(p_all)
    if (nSubplot == 1) {
        ncol <- 1
    } else {
        ncol <- 2
        nrow <- ceiling(nSubplot/2)
        width <- width * ncol
        height <- height * nrow
    }
    # Save
    ggplot2::ggsave(file = targetFile,
        plot = gridExtra::arrangeGrob(grobs = p_all, ncol = ncol),
        device = "png", path = targetFolder, dpi = 100, width = width,
        height = height, units = "cm", limitsize = FALSE)
    if (verbose) { message("Summary EIC plot saved at: ", savePath) }
}
