% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clusterGraph.R
\name{clusterGraph}
\alias{clusterGraph}
\title{Graph-based clustering of cells}
\usage{
clusterGraph(
  x,
  method = c("multilevel", "leiden", "walktrap"),
  multilevel.resolution = 1,
  leiden.resolution = 1,
  leiden.objective = c("modularity", "cpm"),
  walktrap.steps = 4,
  seed = 42
)
}
\arguments{
\item{x}{List containing graph information or an external pointer to a graph, as returned by \code{\link{buildSnnGraph}}.
Alternatively, an \link[igraph]{igraph} object with edge weights.}

\item{method}{String specifying the algorithm to use.
\itemize{
\item \code{"multilevel"} uses multi-level modularity optimization, also known as the Louvain algorithm,
see \url{https://igraph.org/c/doc/igraph-Community.html#igraph_community_multilevel} for details.
\item \code{"walktrap"} uses the Walktrap community finding algorithm,
see \url{https://igraph.org/c/doc/igraph-Community.html#igraph_community_walktrap} for details.
\item \code{"leiden"} uses the Leiden algorithm,
see \url{https://igraph.org/c/doc/igraph-Community.html#igraph_community_leiden} for details.
}}

\item{multilevel.resolution}{Numeric scalar specifying the resolution when \code{method="multilevel"}.
Lower values favor fewer, larger communities; higher values favor more, smaller communities.}

\item{leiden.resolution}{Numeric scalar specifying the resolution when \code{method="leiden"}.
Lower values favor fewer, larger communities; higher values favor more, smaller communities.}

\item{leiden.objective}{String specifying the objective function when \code{method="leiden"}.
\code{"cpm"} uses the Constant Potts Model, which typically yields more fine-grained clusters at the same \code{leiden.resolution}.}

\item{walktrap.steps}{Integer scalar specifying the number of steps to use when \code{method="walktrap"}.
This determines the ability of the Walktrap algorithm to distinguish highly interconnected communities from the rest of the graph.}

\item{seed}{Integer scalar specifying the random seed to use for \code{method="multilevel"} or \code{"leiden"}.}
}
\value{
A list containing \code{membership}, a factor containing the cluster assignment for each cell;
and \code{status}, an integer scalar indicating whether the algorithm completed successfully (0) or not (non-zero).
Additional fields may be present depending on the \code{method}:
\itemize{
\item For \code{method="multilevel"}, the \code{levels} list contains the clustering result at each level of the algorithm.
A \code{modularity} numeric vector also contains the modularity at each level, the highest of which corresponds to the reported \code{membership}.
\item For \code{method="leiden"}, a \code{quality} numeric scalar containg the quality of the partitioning.
\item For \code{method="walktrap"}, a \code{merges} matrix specifies the pair of cells or clusters that were merged at each step of the algorithm.
A \code{modularity} numeric scalar also contains the modularity of the final partitioning.
}
}
\description{
Identify clusters by applying community detection algorithms to a graph.
This assumes that that the nodes on the graph represent cells and weighted edges are formed between related cells.
}
\examples{
data <- matrix(rnorm(10000), ncol=1000)
gout <- buildSnnGraph(data)
str(gout)

str(clusterGraph(gout))
str(clusterGraph(gout, method="leiden"))
str(clusterGraph(gout, method="walktrap"))

}
\seealso{
The various \code{cluster_*} functions in \url{https://libscran.github.io/scran_graph_cluster/}.
}
\author{
Aaron Lun
}
