% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filterClusters.R
\name{filterClusters}
\alias{filterClusters}
\title{Merge clusters and compute all relevant cluster statistics}
\usage{
filterClusters(clusters, highConfSub, coverage, model, genome,
refBase = 'T', minWidth = 12, verbose = TRUE)
}
\arguments{
\item{clusters}{GRanges object containing individual clusters as identified
by the \link{getClusters} function}

\item{highConfSub}{GRanges object containing high-confidence substitution
sites as returned by the \link{getHighConfSub} function}

\item{coverage}{An Rle object containing the coverage at each genomic
position as returned by a call to \link{coverage}}

\item{model}{List of 5 items containing the estimated mixture model as
returned by the \link{fitMixtureModel} function}

\item{genome}{BSgenome object of the relevant reference genome (e.g.
\code{Hsapiens} for the human genome hg19)}

\item{refBase}{A character specifying the base in the reference genome for
which transitions are experimentally induced (e.g. 4-SU treatment - a
standard in PAR-CLIP experiments - induces T to C transitions and hence
\code{refBase = "T"} in this case). Default is "T"}

\item{minWidth}{An integer corresponding to the minimum width of reported
clusters. Shorter clusters are extended to \code{minWidth} starting from the
cluster center}

\item{verbose}{Logical, if TRUE processing steps are printed}
}
\value{
GRanges object containing the transcriptome-wide identified
clusters, having metadata: \item{Ntransitions}{The number of high-confidence
transitions within the cluster} \item{MeanCov}{The mean coverage within the
cluster} \item{NbasesInRef}{The number of genomic positions within the
cluster corresponding to \code{refBase}} \item{CrossLinkEff}{The
crosslinking efficiency within the cluster, estimated as the ratio between
the number of high-confidence transitions within the cluster and the total
number of genomic positions therein corresponding to \code{refBase}}
\item{Sequence}{The genomic sequence undelying the cluster (plus strand)}
\item{SumLogOdds}{The sum of the log-odd values within the cluster}
\item{RelLogOdds}{The sum of the log-odds divided by the number of
high-confidence transitions within the cluster. This variable can be
regarded as a proxy for statistical significance and can be therefore used
to rank clusters. See Comoglio, Sievers and Paro for details.}
}
\description{
If clusters have been identified using the mini-rank norm algorithm, cluster
statistics are computed. In contrast, if the CWT-based cluster
identification algorithm was used, clusters are first filtered to retain
only those instances containing a wavelet peak and a high-confidence
substitution site within their cluster boundaries.
}
\note{
1) This function calls the appropriate processing function according
to the method used to compute clusters. This information is stored in the
\code{metadata(ranges(clusters))} slot as an object of type list.

2) Notice that \code{genome} corresponds to the according reference genome
matching the organism in which experiments have been carried out. For
example \code{genome = Hsapiens} is used for the human reference genome
(assembly 19), where \code{Hsapiens} is provided by
\code{BSgenome.Hsapiens.UCSC.hg19}.
}
\examples{

require(BSgenome.Hsapiens.UCSC.hg19)

data( model, package = "wavClusteR" ) 

filename <- system.file( "extdata", "example.bam", package = "wavClusteR" )
example <- readSortedBam( filename = filename )
countTable <- getAllSub( example, minCov = 10, cores = 1 )
highConfSub <- getHighConfSub( countTable, supportStart = 0.2, supportEnd = 0.7, substitution = "TC" )
coverage <- coverage( example )
clusters <- getClusters( highConfSub = highConfSub, 
                         coverage = coverage, 
                         sortedBam = example, 
	                 cores = 1, 
	                 threshold = 2 ) 

fclusters <- filterClusters( clusters = clusters, 
		             highConfSub = highConfSub, 
        		     coverage = coverage,
			     model = model, 
			     genome = Hsapiens, 
		             refBase = 'T', 
		             minWidth = 12 )
fclusters

}
\references{
Herve Pages, BSgenome: Infrastructure for Biostrings-based
genome data packages

Sievers C, Schlumpf T, Sawarkar R, Comoglio F and Paro R. (2012) Mixture
models and wavelet transforms reveal high confidence RNA-protein interaction
sites in MOV10 PAR-CLIP data, Nucleic Acids Res. 40(20):e160. doi:
10.1093/nar/gks697

Comoglio F, Sievers C and Paro R (2015) Sensitive and highly resolved identification
of RNA-protein interaction sites in PAR-CLIP data, BMC Bioinformatics 16, 32.
}
\seealso{
\code{\link{getClusters}}, \code{\link{getHighConfSub}},
\code{\link{fitMixtureModel}}
}
\author{
Federico Comoglio and Cem Sievers
}
\keyword{core}
