% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AR_drug.R
\name{AR_drug}
\alias{AR_drug}
\title{AR-dependent gene expression in prostate cancer cells}
\format{
\code{MultiAssayExperiment} obtained from an \code{ArchR} project.
Annotated with the Hg38 genome build.
Contains the following experiments:
\itemize{
\item{\strong{TileMatrix}: SingleCellExperiment with 6068436 rows and 23118 columns}
\item{\strong{GeneScoreMatrix}: SingleCellExperiment with 57765 rows and 23118 columns}
\item{\strong{GeneExpressionMatrix}: SingleCellExperiment with 36451 rows and 23118 columns}
\item{\strong{PeakMatrix}: SingleCellExperiment with 237856 rows and 23118 columns}
\item{\strong{MotifMatrix}: SingleCellExperiment with 870 rows and 23118 columns}
\item{\strong{TFPeaksDeviationsMatrix}: SingleCellExperiment with 1533 rows and 23118 columns}
}
}
\usage{
AR_drug(
  metadata = FALSE,
  experiments = c("TileMatrix", "GeneScoreMatrix", "GeneExpressionMatrix", "PeakMatrix",
    "MotifMatrix", "TFPeaksDeviationsMatrix")
)
}
\arguments{
\item{metadata}{logical flag specifying whether to return data or metadata only}

\item{experiments}{character vector of matrices to return; see \code{Format}}
}
\value{
\code{MultiAssayExperiment} made up of \code{SingleCellExperiment}s
with assays stored as \code{DelayedMatrix} objects.
If \code{metadata = TRUE}, an \code{ExperimentHub} object listing this data set's metadata.
}
\description{
Single cell gene expression and ATACseq from prostate cancer cell lines (LNCaP, VCaP, DU145, MDA-PCA-2B, 22Rv1 and NCI-H660).
after 24h of drug treatment (enzalutamide, ARV110 (AR degrader) or SMARCA2_4.1 (SMARCA2/4 degrader) or DMSO control.
}
\section{Data preparation}{


Data can be downloaded Gene Expression Omnibus
(acc. no. \href{https://www.ncbi.nlm.nih.gov/geo/query/acc.cgi?acc=GSE251977}{GSE251977})

Downstream analysis was performed with the \code{ArchR} package:
\subsection{1. Initiate \code{ArchR} project}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# attach ArchR package
library(ArchR)

# configure ArchR
addArchRGenome("hg38")

# create arrow file from fragment files
## list fragment files
fragments <- <FRAGMENT_FILES> # available in Gene Expression Omnibus (GSE251977)
## assign sample names
names(fragments) <- <SAMPLE_IDs>
## create arrows
createArrowFiles(inputFiles = fragments, sampleNames = names(fragments),
                     minTSS = 4, minFrags = 1000)

# specify output directory
outDir <- <OUTPUT_DIRECTORY>

# locate arrow files
arrows <- <ARROW_FILES>
    
doublet.score <- addDoubletScores(
    input = arrows,
    k = 10, #Refers to how many cells near a 'pseudo-doublet' to count.
    knnMethod = 'UMAP', #Refers to the embedding to use for nearest neighbor search.
    LSIMethod = 1
)

# create ArchR project
project <- ArchRProject(arrows, outDir)

}\if{html}{\out{</div>}}
}

\subsection{2. HTO demultiplexing}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(ggplot2)
library(DropletUtils)
library(SingleCellExperiment)
library(Matrix)
library(ArchR)
library(zellkonverter)
library(scater)
library(gridExtra)
library(BiocParallel)


temp_dir <- tempdir()

destfiles_HTO <- c(file.path(temp_dir, "LIB5458339_SAM24418230.hashing.csv"),
                   file.path(temp_dir, "LIB5458340_SAM24418231.hashing.csv"),
                   file.path(temp_dir, "LIB5463784_SAM24425416.hashing.csv"),
                   file.path(temp_dir, "LIB5467656_SAM24427130.hashing.csv")
                   )

download.file(c("https://www.ncbi.nlm.nih.gov/geo/download/?acc=GSE251977&format=file&file=GSE251977\%5FLIB5458339\%5FSAM24418230\%2Ehashing\%2Ecsv\%2Egz",
                "https://www.ncbi.nlm.nih.gov/geo/download/?acc=GSE251977&format=file&file=GSE251977\%5FLIB5458340\%5FSAM24418231\%2Ehashing\%2Ecsv\%2Egz",
                "https://www.ncbi.nlm.nih.gov/geo/download/?acc=GSE251977&format=file&file=GSE251977\%5FLIB5463784\%5FSAM24425416\%2Ehashing\%2Ecsv\%2Egz",
                "https://www.ncbi.nlm.nih.gov/geo/download/?acc=GSE251977&format=file&file=GSE251977\%5FLIB5467656\%5FSAM24427130\%2Ehashing\%2Ecsv\%2Egz"),
              "",
              method = "libcurl"
              )

# download GEX data

destfiles_GEX <- c(file.path(temp_dir, "LIB5458339_SAM24418230_raw_feature_bc_matrix.h5"),
                   file.path(temp_dir, "LIB5458340_SAM24418231_raw_feature_bc_matrix.h5"),
                   file.path(temp_dir, "LIB5463784_SAM24425416_raw_feature_bc_matrix.h5"),
                   file.path(temp_dir, "LIB5467656_SAM24427130_raw_feature_bc_matrix.h5")
                   )

download.file(c(),destfiles_GEX,method = "libcurl")

HTO_uri <- data.frame(SAMID = c("SAM24418230", "SAM24418231", "SAM24425416", "SAM24427130"), HTO_uri = destfiles_HTO)

arcseq_info <-data.frame(SAMID = c("SAM24418230", "SAM24418231", "SAM24425416", "SAM24427130"), uri = destfiles_GEX)

# merge arcseq and HTO paths
merge_file_info <- merge(HTO_uri, arcseq_info, by.x = "SAMID" , by.y = "sampleName")
merge_file_info <- merge_file_info[order(merge_file_info$SAMID), ]



######## load matrices
# load RNA matrix

hashing_qc <- list()
umapplot <- list()
seRNA_final <- list()
for (i in seq_len(nrow(merge_file_info))) \{
    message(merge_file_info$SAMID[i])
    # import gex
    seRNA <- ArchR::import10xFeatureMatrix(
        input = file.path( merge_file_info$uri[i], "raw_feature_bc_matrix.h5"),
        names = merge_file_info$SAMID[i])
    names(assays(seRNA)) <- "counts"

    # import HTO and convert to sce
    HTO <- data.table::fread(merge_file_info$HTO_uri[i])
    rownames_HTO <- HTO$Antibody
    HTO <- HTO[,-1]
    colnames(HTO) <- paste0(merge_file_info$SAMID[i], "#", colnames(HTO),"-1")
    HTO <- as(as.matrix(HTO), "dgCMatrix")
    rownames(HTO) <- rownames_HTO
    HTO <- SingleCellExperiment(assays = list(counts=HTO))

    # merge GEX and HTO into a SCE
    common_cells <- intersect(colnames(HTO), colnames(seRNA))
    HTO <- HTO[, common_cells]
    seRNA <- seRNA[, common_cells]
    seRNA <- as(seRNA, "SingleCellExperiment")
    altExp(seRNA, "HTO") <- HTO

    # call empty droplets to define ambient droplets
    set.seed(10010)
    e.out.gene <- emptyDrops(counts(seRNA), by.rank = 30000 )
    is.cell <- e.out.gene$FDR <= 0.001
    summary(is.cell)

    # plot empty droplet assignments
    par(mfrow=c(1,2))
    r <- rank(-e.out.gene$Total)
    plot(r, e.out.gene$Total, log="xy", xlab="Rank", ylab="Total gene count", main="")
    abline(h=metadata(e.out.gene)$retain, col="darkgrey", lty=2, lwd=2)
    hist(log10(e.out.gene$Total[is.cell]), xlab="Log[10] gene count", main="")

    # Estimate HTO ambient proportions using empty droplets
    hto.mat <- assay(altExp(seRNA),"counts")[,which(is.cell)]
    ambient <- proportions(rowSums(assay(altExp(seRNA), "counts")[,is.na(e.out.gene$FDR)]))
    # plot ambient proportions
    barplot(ambient,las=2, main="ambient proportion")
    hash.stats <- hashedDrops(hto.mat, ambient=ambient)
    table(hash.stats$Best[hash.stats$Confident])

    # examine hashing
    colors <- rep("grey", nrow(hash.stats))
    colors[hash.stats$Doublet] <- "red"
    colors[hash.stats$Confident] <- "black"

    hashing_qc[[merge_file_info$SAMID[i]]] <- plot(hash.stats$LogFC, hash.stats$LogFC2,
                                                   xlab="Log fold-change from best to second HTO",
                                                   ylab="Log fold-change of second HTO over ambient",
                                                   col=colors,
                                                   main=merge_file_info$SAMID[i])

    # keep only non-empty cells
    seRNA <- seRNA[, which(is.cell)]
    colData(seRNA) <- cbind(colData(seRNA), hash.stats)
    colData(seRNA)$library <- sapply(strsplit(colnames(seRNA), split = "#"), "[",1)

    assay(altExp(seRNA), "logcounts") <- log10(assay(altExp(seRNA), "counts")+1)
    assay(altExp(seRNA), "clr") <- sweep(assay(altExp(seRNA), "logcounts"), 2,
                                         colMeans(assay(altExp(seRNA), "logcounts")), "-")
    seRNA <- runUMAP(seRNA, altexp = "HTO", name="UMAP_HTO", assay.type = "clr", exprs_values = "clr")
    seRNA$hash_assignment <- rownames_HTO[seRNA$Best]
    umapplot[[merge_file_info$SAMID[i]]] <- plotReducedDim(seRNA[, which(seRNA$Doublet == FALSE & seRNA$Confident == TRUE)],
                                                           dimred = "UMAP_HTO",
                                                           color_by = "hash_assignment",
                                                           point_size=0.5, rasterise=TRUE) +
        ggtitle(merge_file_info$SAMID[i])

    # save seRNA
    seRNA_final[[merge_file_info$SAMID[i]]] <- seRNA
\}

seRNA_final <- do.call(cbind, seRNA_final)


library(ArchR)
seRNA_final$hash_assignment2 <- paste0(seRNA_final$library, seRNA_final$hash_assignment)


# copy ArchR project from /gstore/data/genomics/congee_rest_runs/6679b364f79a145e53521a61/ArchR_output/croo_output/ArchRProject
proj <- ArchR::loadArchRProject("OUTPUT/ArchRProject/")
common <- intersect(proj$cellNames, colnames(seRNA_final))
proj <- proj[common,]

# add HTO information

for (row_data in colnames(colData(seRNA_final)))\{
    proj <- addCellColData(
        ArchRProj = proj,
        data = colData(seRNA_final)[common, row_data],
        cells = common,
        name = row_data,
        force = TRUE
    )
\}

#filter out doublets and non-confident calls
proj <- proj[which(proj$Confident == TRUE & proj$Doublet == FALSE), ]
}\if{html}{\out{</div>}}
}

\subsection{3. Clustering}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{
# filter doublets
archr.proj <- filterDoublets(
    ArchRProj = archr.proj,
    cutEnrich = 1,
    cutScore = -Inf,
    filterRatio = 1
)

# filter cells that do not contain rna
archr.proj <- archr.proj[!is.na(archr.proj$Gex_nUMI)]

# add reduced dims
archr.proj <- addIterativeLSI(
    ArchRProj = archr.proj,
    useMatrix = 'TileMatrix',
    name = 'IterativeLSI_TileMatrix',
    threads = 4,
    seed = 2,
    force = TRUE
)

archr.proj <- addIterativeLSI(
    ArchRProj = archr.proj,
    useMatrix = 'GeneExpressionMatrix',
    name = 'IterativeLSI_GeneExpressionMatrix',
    firstSelection = "variable",
    depthCol = "Gex_nUMI",
    varFeatures = 2500,
    binarize = FALSE,
    threads = 4,
    seed = 2,
    force = TRUE
)

archr.proj <- addCombinedDims(
    archr.proj,
    reducedDims = c('IterativeLSI_TileMatrix', 'IterativeLSI_GeneExpressionMatrix'),
    name =  'IterativeLSI_Combined'
)


# add clusters
archr.proj <- addClusters(
input = archr.proj,
reducedDims = 'IterativeLSI_Combined',
name = 'Clusters_Combined',
seed = 2,
prefix = 'Combined_C',
force = TRUE,
method = "scran"
)
cM <- confusionMatrix(getCellColData(archr.proj)[,'hash_assignment2'], archr.proj$Sample)
cM <- cM / Matrix::rowSums(cM)
pheatmap::pheatmap(
    mat = as.matrix(cM),
    color = paletteContinuous('whiteBlue'),
    border_color = 'black',
    filename = paste0(getOutputDirectory(archr.proj),'/Plots/GPSA-Sample-hash_assignment2-pheatmap.pdf')
)


# add embeddings

archr.proj <- addUMAP(
    ArchRProj = archr.proj,
    reducedDims = 'IterativeLSI_Combined',
    name = 'UMAP_Combined',
    seed = 2,
    threads = 1,
    force = TRUE
)

archr.proj <- addUMAP(
    ArchRProj = archr.proj,
    reducedDims = 'IterativeLSI_TileMatrix',
    name = 'UMAP_ATAC',
    seed = 2,
    threads = 1,
    force = TRUE
)


archr.proj <- addUMAP(
    ArchRProj = archr.proj,
    reducedDims = 'IterativeLSI_GeneExpressionMatrix',
    name = 'UMAP_RNA',
    seed = 2,
    threads = 1
)



# tSNE
archr.proj <- addTSNE(
    ArchRProj = archr.proj,
    reducedDims = 'IterativeLSI_Combined',
    name = 'TSNE_Combined',
    perplexity = 30,
    seed = 2,
    force = TRUE,
    threads = max(floor(4/2), 1)
)

}\if{html}{\out{</div>}}
}

\subsection{4. Clustering and motif annotation}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Peak calling

library(BSgenome.Hsapiens.Genentech.GRCh38)

archr.proj <- addGroupCoverages(
    ArchRProj = archr.proj,
    groupBy = 'hash_assignment2',
    threads = 4
)


archr.proj <- ArchR.helper::addReproduciblePeakSet(
    ArchRProj = archr.proj,
    groupBy = 'hash_assignment2',
    peakMethod = "MACSr",
    excludeChr = c('chrMT','chrY'),
    genomeSize = 2.7e9,
    threads = 4,
    force = TRUE
)

archr.proj <- addPeakMatrix(
    ArchRProj = archr.proj,
    binarize = FALSE,
    threads = 1,
    force = TRUE
)

# TF annotation
peaks.anno <- scMultiome::tfBinding()
archr.proj <- addPeakAnnotations(
    ArchRProj = archr.proj,
    regions = peaks.anno,
    name = 'TF_peaks',
    force = TRUE
)

archr.proj <- addDeviationsMatrix(
    ArchRProj = archr.proj,
    peakAnnotation = 'TF_peaks',
    matrixName = 'TFPeaksDeviationsMatrix',
    threads = 1,
    force = TRUE
)


# motif annotation
archr.proj  <- addMotifAnnotations(ArchRProj = archr.proj, motifSet = 'cisbp', name = 'Motif', species='Homo sapiens')

archr.proj <- addDeviationsMatrix(
    ArchRProj = archr.proj,
    peakAnnotation = 'Motif',
    threads = 1,
    force = TRUE
)



#add BigWigs

getGroupBW(
    ArchRProj = archr.proj,
    groupBy = "hash_assignment2",
    normMethod = "ReadsInTSS",
    threads = 1,
)


getGroupBW(
    ArchRProj = archr.proj,
    groupBy = 'hash_assignment2',
    normMethod = 'ReadsInTSS',
    threads = 1
)

#add extra  cell information
sample_info <- read.csv("HTO_SAMID.csv") # to be dowloaded from Gene Expression Omnibus, GSE251977
sample_info$SAMID_HTO <- paste0(sample_info$SAMID, sample_info$HTO)

archr.proj$TREATMENT <- sample_info$TREATMENT[match(archr.proj$hash_assignment2, sample_info$SAMID_HTO)]
archr.proj$Cell <- unlist(lapply(strsplit(archr.proj$TREATMENT, split = "-"),"[",1))
archr.proj$TEST_ARTICLE <- unlist(lapply(strsplit(archr.proj$TREATMENT, split = "-"),"[",2))

archr.proj$Cell[archr.proj$Cell == "22RV1"] <- "22Rv1"

}\if{html}{\out{</div>}}
}

\subsection{5. Save results}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# convert project to MultiAssayExperiment object
MAE <- maw.archr::create.mae.with.multiple.sces.from.archr(outDir)

# save object
saveMAE("inst/extdata/prostateENZ.h5")
}\if{html}{\out{</div>}}
}
}

\section{Data storage and access}{

The \code{MultiAssayExperiments} is split into separate \code{SingleCellExperiment}
objects and they in turn are split into components, all of which are stored in a
single hdf5 file. Data and can be accessed with a special function that extracts
elements of the requested experiment(s), reassembles them, and builds an MAE.
}

\examples{
# check metada of dataset
AR_drug(metadata = TRUE)

# download data
 \dontrun{
AR_drug()
}

}
\references{
Tomasz Włodarczyk, Aaron Lun, Diana Wu, Shreya Menon, Shushan Toneyan, Kerstin Seidel, Liang Wang,
Jenille Tan, Shang-Yang Chen, Timothy Keyes, Aleksander Chlebowski, Yu Guo, Ciara Metcalfe,
Marc Hafner, Christian W. Siebel, M. Ryan Corces, Robert Yauch, Shiqi Xie, Xiaosai Yao. 2023.
"Inference of single-cell transcription factor activity to dissect mechanisms of lineage
plasticity and drug response" bioRxiv 2023.11.27.568955; doi: https://doi.org/10.1101/2023.11.27.568955
}
