% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vis_gene.R
\name{vis_gene}
\alias{vis_gene}
\title{Sample spatial gene visualization}
\usage{
vis_gene(
  spe,
  sampleid = unique(spe$sample_id)[1],
  geneid = rowData(spe)$gene_search[1],
  spatial = TRUE,
  assayname = "logcounts",
  minCount = 0,
  viridis = TRUE,
  image_id = "lowres",
  alpha = NA,
  cont_colors = if (viridis) viridisLite::viridis(21) else c("aquamarine4",
    "springgreen", "goldenrod", "red"),
  point_size = 2,
  auto_crop = TRUE,
  na_color = "#CCCCCC40",
  multi_gene_method = c("z_score", "pca", "sparsity"),
  is_stitched = FALSE,
  cap_percentile = 1,
  ...
)
}
\arguments{
\item{spe}{A
\link[SpatialExperiment:SpatialExperiment]{SpatialExperiment-class}
object. See \code{\link[=fetch_data]{fetch_data()}} for how to download some example objects or
\code{\link[=read10xVisiumWrapper]{read10xVisiumWrapper()}} to read in \code{spaceranger --count} output files and
build your own \code{spe} object.}

\item{sampleid}{A \code{character(1)} specifying which sample to plot from
\code{colData(spe)$sample_id} (formerly \code{colData(spe)$sample_name}).}

\item{geneid}{A \code{character()} specifying the gene ID(s) stored in
\code{rowData(spe)$gene_search} or a continuous variable(s) stored in
\code{colData(spe)} to visualize. For each ID, if \code{rowData(spe)$gene_search} is
missing, then \code{rownames(spe)} is used to search for the gene ID. When a
vector of length > 1 is supplied, the continuous variables are combined
according to \code{multi_gene_method}, producing a single value for each spot.}

\item{spatial}{A \code{logical(1)} indicating whether to include the histology
layer from \code{\link[=geom_spatial]{geom_spatial()}}. If you plan to use
\link[plotly:ggplotly]{ggplotly()} then it's best to set this to \code{FALSE}.}

\item{assayname}{The name of the \code{assays(spe)} to use for extracting the
gene expression data. Defaults to \code{logcounts}.}

\item{minCount}{A \code{numeric(1)} specifying the minimum gene expression (or
value in the continuous variable) to visualize. Values at or below this
threshold will be set to \code{NA}. Defaults to \code{0}.}

\item{viridis}{A \code{logical(1)} whether to use the color-blind friendly
palette from \link[viridisLite:viridis]{viridis} or the color palette used
in the paper that was chosen for contrast when visualizing the data on
top of the histology image. One issue is being able to differentiate low
values from NA ones due to the purple-ish histology information that is
dependent on cell density.}

\item{image_id}{A \code{character(1)} with the name of the image ID you want to
use in the background.}

\item{alpha}{A \code{numeric(1)} in the \verb{[0, 1]} range that specifies the
transparency level of the data on the spots.}

\item{cont_colors}{A \code{character()} vector of colors that supersedes the
\code{viridis} argument.}

\item{point_size}{A \code{numeric(1)} specifying the size of the points. Defaults
to \code{1.25}. Some colors look better if you use \code{2} for instance.}

\item{auto_crop}{A \code{logical(1)} indicating whether to automatically crop
the image / plotting area, which is useful if the Visium capture area is
not centered on the image and if the image is not a square.}

\item{na_color}{A \code{character(1)} specifying a color for the NA values.
If you set \code{alpha = NA} then it's best to set \code{na_color} to a color that has
alpha blending already, which will make non-NA values pop up more and the NA
values will show with a lighter color. This behavior is lost when \code{alpha} is
set to a non-\code{NA} value.}

\item{multi_gene_method}{A \code{character(1)}: either \code{"pca"}, \code{"sparsity"}, or
\code{"z_score"}. This parameter controls how multiple continuous variables are
combined for visualization, and only applies when \code{geneid} has length
great than 1. \code{z_score}: to summarize multiple continuous variables, each is
normalized to represent a Z-score. The multiple scores are then averaged.
\code{pca}: PCA dimension reduction is conducted on the matrix formed by the
continuous variables, and the first PC is then used and multiplied by -1 if
needed to have the majority of the values for PC1 to be positive. \code{sparsity}:
the proportion of continuous variables with positive values for each spot is
computed. For more details, check the multi gene vignette at
\url{https://research.libd.org/spatialLIBD/articles/multi_gene_plots.html}.}

\item{is_stitched}{A \code{logical(1)} vector: If \code{TRUE}, expects a
\link[SpatialExperiment:SpatialExperiment]{SpatialExperiment-class} built
with \code{visiumStitched::build_spe()}.
\url{http://research.libd.org/visiumStitched/reference/build_spe.html}; in
particular, expects a logical colData column \code{exclude_overlapping}
specifying which spots to exclude from the plot. Sets \code{auto_crop = FALSE}.}

\item{cap_percentile}{A \code{numeric(1)} in (0, 1] determining the maximum
percentile (as a proportion) at which to cap expression. For example, a value
of 0.95 sets the top 5\% of expression values to the 95th percentile value.
This can help make the color scale more dynamic in the presence of high
outliers. Defaults to \code{1}, which effectively performs no capping.}

\item{...}{Passed to \link[base:paste]{paste0()} for making the title of the
plot following the \code{sampleid}.}
}
\value{
A \link[ggplot2:ggplot]{ggplot2} object.
}
\description{
This function visualizes the gene expression stored in \code{assays(spe)} or any
continuous variable stored in \code{colData(spe)} for one given sample at the
spot-level using (by default) the histology information on the background.
To visualize clusters (or any discrete variable) use \code{\link[=vis_clus]{vis_clus()}}.
}
\details{
This function subsets \code{spe} to the given sample and prepares the
data and title for \code{\link[=vis_gene_p]{vis_gene_p()}}. It also adds a caption to the plot.
}
\examples{

if (enough_ram()) {
    ## Obtain the necessary data
    if (!exists("spe")) spe <- fetch_data("spe")

    ## Valid `geneid` values are those in
    head(rowData(spe)$gene_search)
    ## or continuous variables stored in colData(spe)
    ## or rownames(spe)

    ## Visualize a default gene on the non-viridis scale
    p1 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        viridis = FALSE
    )
    print(p1)

    ## Use a custom set of colors in the reverse order than usual
    p2 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        cont_colors = rev(viridisLite::viridis(21, option = "magma"))
    )
    print(p2)

    ## Turn the alpha to 1, which makes the NA values have a full alpha
    p2b <- vis_gene(
        spe = spe,
        sampleid = "151507",
        cont_colors = rev(viridisLite::viridis(21, option = "magma")),
        alpha = 1
    )
    print(p2b)

    ## Turn the alpha to NA, and use an alpha-blended "forestgreen" for
    ## the NA values
    # https://gist.githubusercontent.com/mages/5339689/raw/2aaa482dfbbecbfcb726525a3d81661f9d802a8e/add.alpha.R
    # add.alpha("forestgreen", 0.5)
    p2c <- vis_gene(
        spe = spe,
        sampleid = "151507",
        cont_colors = rev(viridisLite::viridis(21, option = "magma")),
        alpha = NA,
        na_color = "#228B2280"
    )
    print(p2c)

    ## Visualize a continuous variable, in this case, the ratio of chrM
    ## gene expression compared to the total expression at the spot-level
    p3 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        geneid = "expr_chrM_ratio"
    )
    print(p3)

    ## Visualize a gene using the rownames(spe)
    p4 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        geneid = rownames(spe)[which(rowData(spe)$gene_name == "MOBP")]
    )
    print(p4)

    ## Repeat without auto-cropping the image
    p5 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        geneid = rownames(spe)[which(rowData(spe)$gene_name == "MOBP")],
        auto_crop = FALSE
    )
    print(p5)

    #    Define several markers for white matter
    white_matter_genes <- c(
        "ENSG00000197971", "ENSG00000131095", "ENSG00000123560",
        "ENSG00000171885"
    )

    ## Plot all white matter markers at once using the Z-score combination
    ## method. Flatten this quantity at the top 5\% of values for plotting
    p6 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        geneid = white_matter_genes,
        multi_gene_method = "z_score",
        cap_percentile = 0.95
    )
    print(p6)

    ## Plot all white matter markers at once using the sparsity combination
    ## method
    p7 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        geneid = white_matter_genes,
        multi_gene_method = "sparsity"
    )
    print(p7)

    ## Plot all white matter markers at once using the PCA combination
    ## method
    p8 <- vis_gene(
        spe = spe,
        sampleid = "151507",
        geneid = white_matter_genes,
        multi_gene_method = "pca"
    )
    print(p8)
}
}
\seealso{
Other Spatial gene visualization functions: 
\code{\link{vis_gene_p}()},
\code{\link{vis_grid_gene}()}
}
\concept{Spatial gene visualization functions}
